( function() {
	'use strict';
	const siteNavigations = document.querySelectorAll("#primary-navigation, #secondary-navigation, #vertical-menu");
	
	// Return early if the navigation don't exist.
	if ( ! siteNavigations ) {
		return;
	}
	siteNavigations.forEach(function (siteNavigation, index) {
		const menu = siteNavigation.getElementsByTagName( 'ul' )[ 0 ];

		// Get all the link elements within the menu.
		const links = menu.getElementsByTagName( 'a' );
		
		// Get all the link elements with children within the menu.
		const linksWithChildren = menu.querySelectorAll( '.menu-item-has-children > a, .page_item_has_children > a' );
		
		// Toggle focus each time a menu link is focused or blurred.
		for ( const link of links ) {
			link.addEventListener( 'focus', toggleFocus, { passive: true, capture: true } );
			link.addEventListener( 'blur', toggleFocus, { passive: true, capture: true } );
		}

		// Toggle focus each time a menu link with children receive a touch event.
		for ( const link of linksWithChildren ) {
			link.addEventListener( 'touchstart', toggleFocus, { passive: true } );
		}
	});
	/**
	 * Sets or removes .focus class on an element.
	 */
	function toggleFocus() {
		if ( event.type === 'focus' || event.type === 'blur' ) {
			let self = this;
			// Move up through the ancestors of the current link until we hit .nav-menu.
			while ( ! self.classList.contains( 'nevo-nav-menu' ) ) {
				// On li elements toggle the class .focus.
				if ( 'li' === self.tagName.toLowerCase() ) {
					self.classList.toggle( 'focus' );
				}
				self = self.parentNode;
			}
		}

		if ( event.type === 'touchstart' ) {
			const menuItem = this.parentNode;
			event.preventDefault();
			for ( const link of menuItem.parentNode.children ) {
				if ( menuItem !== link ) {
					link.classList.remove( 'focus' );
				}
			}
			menuItem.classList.toggle( 'focus' );
		}
	}
	window.nevoHelper = {
		/**
		 * Helper function to get element's offset.
		 */
		getOffset: function( $el ) {
			if ( $el instanceof HTMLElement ) {
				var rect = $el.getBoundingClientRect();

				return {
					top: rect.top + window.pageYOffset,
					left: rect.left + window.pageXOffset,
				}
			}

			return {
				top: null,
				left: null,
			};
		},

		/**
		 * Helper function to check if element's visible or not.
		 */
		isVisible: function( $el ) {
			return $el.offsetWidth > 0 && $el.offsetHeight > 0;
		},

		/**
		 * Function to check RTL
		 */
		isRTL: function() {
			return document.body.classList.contains( 'rtl' );
		},

	}
	window.nevo = {
		/**
		 * Function to init page popup toggle.
		 */
		initGlobalPopup: function() {
			var $clickedToggle = null;
			var $modal = null;
			var $menuItems = null;
			var $closeButton = null;

			var deactivatePopup = function() {
				var $activePopups = Array.prototype.slice.call(document.querySelectorAll('.nevo-popup-active'));

				$activePopups.forEach(function($activePopup) {
					// Deactivate popup.
					$clickedToggle.classList.remove('nevo-popup-toggle-active');
					$clickedToggle.setAttribute('aria-expanded', false);
					$activePopup.classList.remove('nevo-popup-active');
					document.body.classList.remove('nevo-has-popup-active');

					// Back current focus to the toggle.
					$activePopup.removeAttribute('tabindex');
					if (document.body.classList.contains('using-keyboard')) {
						$clickedToggle.focus();
					}
				});
			};

			// Show / hide popup when the toggle is clicked.
			var handlePopupToggle = function(e) {
				// Check target element.
				var $this = e.target.closest('.nevo-popup-toggle');
				if (!$this) return;

				e.preventDefault();

				var $target = document.querySelector('#' + $this.getAttribute('data-target'));

				// Abort if no popup target found.
				if (!$target) return;

				if ($target.classList.contains('nevo-popup-active')) {
					deactivatePopup();
				} else {
					// Activate popup.
					$this.classList.add('nevo-popup-toggle-active');
					$this.setAttribute('aria-expanded', true);
					$target.classList.add('nevo-popup-active');
					document.body.classList.add('nevo-has-popup-active');

					// Put focus on popup.
					setTimeout(function() {
						$target.setAttribute('tabindex', 0);
						$target.focus();
						// Get all menu items and the close button inside the modal
						$menuItems = Array.prototype.slice.call($target.querySelectorAll('.menu-item'));
						$closeButton = $target.querySelector('.close-button');
					}, 300);

					// Save this toggle and modal for putting back focus when popup is deactivated.
					$clickedToggle = $this;
					$modal = $target;
				}
			};

			document.addEventListener('click', handlePopupToggle, false);
			document.addEventListener('touchend', handlePopupToggle, false);

			// Close popup when any of ".popup-close" element is clicked.
			var handlePopupClose = function(e) {
				// Check target element.
				if (!e.target.closest('.popup-close')) return;

				e.preventDefault();

				deactivatePopup();
			};

			document.addEventListener('click', handlePopupClose, false);
			document.addEventListener('touchend', handlePopupClose, false);

			// Close popup using "escape" keyboard button.
			var handlePopupEscape = function(e) {
				var key = e.which || e.keyCode;

				if ($modal && $modal.classList.contains('nevo-popup-active') && 27 === key) {
					deactivatePopup();
				}
			};

			document.addEventListener('keydown', handlePopupEscape, false);

			// When window resize, close Active Popups based on their responsive visibility classes.
			var handleResponsiveVisibility = function(e) {
				if ($modal && $modal.classList.contains('nevo-popup-active')) {
					var device = 'mobile';

					if (Nevo_JS.breakpoints.mobile <= window.innerWidth) {
						device = 'tablet';
					}

					if (Nevo_JS.breakpoints.desktop <= window.innerWidth) {
						device = 'desktop';
					}

					deactivatePopup(device);
				}
			};

			window.addEventListener('resize', handleResponsiveVisibility, false);

			// Close popup if any hash link is clicked.
			var handleHashLinkInsidePopup = function(e) {
				// Check target element.
				if (!e.target.closest('.nevo-popup a')) return;

				var $link = e.target.closest('a');

				// Check if the link is a hash link.
				if ('' !== $link.hash) {
					var pageURL = (window.location.hostname + '/' + window.location.pathname).replace(/\/$/, ''),
						linkURL = ($link.hostname + '/' + $link.pathname).replace(/\/$/, '');

					// Check if the hash target is on this page.
					if (pageURL === linkURL) {
						// Deactivate all popups.
						if ($modal && $modal.classList.contains('nevo-popup-active')) {
							deactivatePopup();
						}
					}
				}
			};

			document.addEventListener('click', handleHashLinkInsidePopup, false);
			document.addEventListener('touchend', handleHashLinkInsidePopup, false);

			// Trap focus inside the modal
			var trapFocusInsideModal = function(e) {
				var key = e.which || e.keyCode;

				if ($modal && $modal.classList.contains('nevo-popup-active') && (key === 9)) {
					// Tab key pressed
					var focusableElements = $modal.querySelectorAll(
						'a, button, input, select, textarea, [tabindex]:not([tabindex="-1"])'
					);

					var firstElement = focusableElements[0];
					var lastElement = focusableElements[focusableElements.length - 1];

					if (!e.shiftKey && document.activeElement === lastElement) {
						e.preventDefault();
						firstElement.focus();
					} else if (e.shiftKey && document.activeElement === firstElement) {
						e.preventDefault();
						lastElement.focus();
					}
				}
			};

			document.addEventListener('keydown', trapFocusInsideModal, false);
		},
		/**
		 * Function to init toggle menu.
		 */
		initToggleDropdown: function() {
			var $clickedToggle = null;
			
			var handleSubMenuToggle = function( e ) {
				// Check target element.
				var $this = e.target.closest( '.nevo-dropdown .nevo-toggle' );
				if ( ! $this ) return;
				
				e.preventDefault();

				var $header = document.getElementById( 'masthead' ),
				    $menuItem = $this.parentElement;

				// Menu item already has "focus" class, so collapses itself.
				if ( $menuItem.classList.contains( 'focus' ) ) {
					$menuItem.classList.remove( 'focus' );
					$this.setAttribute( 'aria-expanded', false );
				}
				// Menu item doesn't have "focus" class yet, so collapses other focused menu items found in the header and focuses this menu item.
				else {
					var $focusedMenuItems = Array.prototype.slice.call( $header.querySelectorAll( '.menu-item.focus' ) );
					$focusedMenuItems.forEach(function( $focusedMenuItem ) {
						$focusedMenuItem.classList.remove( 'focus' );
					});
					$menuItem.classList.add( 'focus' );
					$this.setAttribute( 'aria-expanded', true );
					// Save this toggle for putting back focus when popup is deactivated.
					$clickedToggle = $this;
				}
			}
			document.addEventListener( 'click', handleSubMenuToggle, false );
			document.addEventListener( 'touchend', handleSubMenuToggle, false );
			
			/**
			 * Empty Hash Link Handler
			 */
			var handleAccordionMenuEmptyHashLink = function( e ) {
				// Check target element.
				var $this = e.target.closest( '.nevo-dropdown .menu-item-has-children > .nevo-menu-link[href="#"]' );
				if ( ! $this ) return;

				e.preventDefault();

				var $menuItem = $this.parentElement,
				    $toggle = $menuItem.querySelector( '.nevo-toggle' );

				// If an empty hash link is clicked, trigger the toggle click event.
				$toggle.click();
			}
			document.addEventListener( 'click', handleAccordionMenuEmptyHashLink, false );
			document.addEventListener( 'touched', handleAccordionMenuEmptyHashLink, false );
		},
		
		/**
		 * Function to init scroll to top.
		 */
		initScrollToTop: function() {
			var $scrollToTop = document.querySelector( '.nevo-scroll-to-top' );

			if ( $scrollToTop ) {
				var handleScrollToTop = function( e ) {
					// Check target element.
					if ( ! e.target.closest( '.nevo-scroll-to-top' ) ) return;

					e.preventDefault();

					var $link = e.target.closest( '.nevo-scroll-to-top' ),
					    $target = document.getElementById( $link.getAttribute( 'href' ).replace( '#', '' ) );

					if ( $target ) {
						window.scrollTo({
							top: $target.getBoundingClientRect().top,
							behavior: 'smooth',
						});
					}
				}
				document.addEventListener( 'click', handleScrollToTop, false );
				document.addEventListener( 'touchend', handleScrollToTop, false );

				if ( $scrollToTop.classList.contains( 'scrolltop-display-sticky' ) ) {
					var checkStickyOffset = function() {
						if ( window.pageYOffset > 0.5 * window.innerHeight ) {
							$scrollToTop.classList.add( 'nevo-sticky' );
						} else {
							$scrollToTop.classList.remove( 'nevo-sticky' );
						}
					}
					window.addEventListener( 'scroll', checkStickyOffset, false );
					checkStickyOffset();
				}
			}
		},
		// Function to check if lists have markers (numbers or bullets)
		hasMarker: function() {
		  // Add your JavaScript code here:
		  // Get all ul and ol elements on the page
		  const lists = document.querySelectorAll('ul, ol');

		  // Iterate through each list element
		  lists.forEach((list) => {
			// Check if the list contains list items with numbers or bullets
			const hasMarker = list.querySelector('li') && getComputedStyle(list.querySelector('li'))['list-style-type'] !== 'none';

			// If the list has list items with numbers or bullets, add the 'has-marker' class to it
			if (hasMarker) {
			  list.classList.add('has-marker');
			} else {
			  // If the list doesn't have list items with numbers or bullets, add the 'no-marker' class to avoid indentation
			  list.classList.add('no-marker');
			}
		  });
		},
		
		/**
		 * Function that calls all init functions.
		 */
		initAll: function() {
			window.nevo.initGlobalPopup();
			window.nevo.initToggleDropdown();
			window.nevo.initScrollToTop();
			window.nevo.hasMarker();
		},
	}

	document.addEventListener( 'DOMContentLoaded', window.nevo.initAll, false );
}() );