<?php
/**
 * Comment Shortcode
 *
 * This file is a part of the Nevo and should not be edited directly.
 * Instead, all customizations should be done through Nevo Child Themes.
 *
 * @author CaoNguyen
 * @package Nevo
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Callback for the display-comments shortcode.
 *
 * @param array $atts Shortcode attributes.
 * @return string
 */
function nevo_display_comments_shortcode( $atts ) {
	/**
	 * Short circuit filter.
	 *
	 * Use this filter to return from this function immediately, with the return of the filter callback.
	 *
	 * @since 1.0
	 *
	 * @param bool  $short_circuit False to allow this function to continue, anything else to return that value.
	 * @param array $atts          Shortcode attributes.
	 */
	$atts = shortcode_atts(
		array(
			'post_type'     => 'post',
			'limit'         => '5',
			'offset'        => '',
			'order'         => 'DESC',
			'exclude_pings' => 0,
			'avatar'        => true,
			'avatar_size'   => 55,
			'avatar_type'   => 'rounded-circle',
			'avatar_align'  => 'alignleft',
			'time'          => true,
			'excerpt'       => true,
			'excerpt_limit' => 50,
			'status'        => 'approve',
			'post_status'   => 'publish',
		),
		$atts,
		'display-comments'
	);
	
	$post_type     = sanitize_text_field( $atts['post_type'] );
	$limit         = (int) $atts['limit'];
	$offset        = (int) $atts['offset'];
	$order         = sanitize_key( $atts['order'] );
	$exclude_pings = (int) $atts['exclude_pings'];
	$avatar        = filter_var( $atts['avatar'], FILTER_VALIDATE_BOOLEAN );
	$avatar_size   = sanitize_text_field( $atts['avatar_size'] );
	$avatar_type   = array_map( 'sanitize_html_class', explode( ' ', $atts['avatar_type'] ) );
	$avatar_align  = array_map( 'sanitize_html_class', explode( ' ', $atts['avatar_align'] ) );
	$time          = filter_var( $atts['time'], FILTER_VALIDATE_BOOLEAN );
	$excerpt       = filter_var( $atts['excerpt'], FILTER_VALIDATE_BOOLEAN );
	$excerpt_limit = sanitize_text_field( $atts['excerpt_limit'] );
	$status        = sanitize_text_field( $atts['status'] );
	$post_status   = sanitize_text_field( $atts['post_status'] );
	
	$query = array(
		'number'      => $limit,
		'offset'      => $offset,
		'order'       => $order,
		'post_status' => $post_status,
		'post_type'   => nevo_list_explode( $post_type ),
		'status'      => $status
	);

	if ( 1 === $exclude_pings ) {
		$query['type__not_in'] = 'pings';
	}
	
	// Allow plugins/themes developer to filter the default comment query.
	$query = apply_filters( 'nevo_comments_args', $query );

	// Get the comments.
	$comments = get_comments( $query );
	
	// Set up a default, empty variable.
	$html = '';
	$date_format = get_option( 'date_format' );
	$time_format = '';
	
	if ( apply_filters( 'nevo_show_comment_time_format', false ) ) {
		$time_format = get_option( 'time_format' );
	}
					
	if ( is_array( $comments ) && $comments ) :

		$html = '<ul class="cv-recent-comment-ul">';

			foreach( $comments as $comment ) :

				$html .= '<li class="recent-comments nevo-recent-comment-li">';

					if ( $avatar ) :
						$html .= '<div class="comment-avatar '. implode( ' ', $avatar_align ) .'">' . get_avatar( $comment->comment_author_email, $avatar_size, false, false, array( 'class' => implode( ' ', $avatar_type ) ) ) . '</div>';
					endif;
					
					$html .= '<div class="comments-meta d-block"><span class="comment-user d-inline-block text-capitalize">'. get_comment_author_link( $comment->comment_ID ) .'</span>';
					if ( $time ) :
						if ( apply_filters( 'nevo_show_comment_time_ago', true ) ) :
							$html .= '<time class="d-inline-block entry-meta cv-ms-3 small" datetime="'. date_i18n( $date_format . ' ' . $time_format, strtotime( $comment->comment_date ) ) .'" itemprop="datePublished">';
							$html .= sprintf( esc_html__( '%s ago', 'nevo' ), human_time_diff( strtotime( $comment->comment_date ), current_time( 'timestamp' ) ) );
							$html .= '</time>';
						else:
							$html .= '<time class="d-inline-block entry-meta cv-ms-3 small" datetime="'. date_i18n( $date_format . ' ' . $time_format, strtotime( $comment->comment_date ) ) .'" itemprop="datePublished">'. date_i18n( $date_format . ' ' . $time_format, strtotime( $comment->comment_date ) ) .'</time>';
						endif;
					endif;
					$html .= '</div>';
						
					$html .= sprintf( __( '%1$s On %3$s %2$s ', 'nevo' ),
						'<div class="comment-on d-block">',
						'</div>',
						'<a class="comment-link" href="'. esc_url( get_comment_link( $comment->comment_ID ) ) .'">' . get_the_title( $comment->comment_post_ID ) . '</a>'
					);

					if ( $excerpt ) :
						$html .= '<span class="comment-excerpt d-block">' . wp_html_excerpt( $comment->comment_content, absint( $excerpt_limit ), '&hellip;' ) . '</span>';
					endif;

				$html .= '</li>';

			endforeach;

		$html .= '</ul>';

	endif;

	// Return the comments markup.
	return $html;
}