<?php
/**
 *  Helpers functions.
 *
 * @package Nevo
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'nevo_get_template_part' ) ) {
	/**
	 * Modified version of the original `get_template_part` function.
	 * Add filters to allow 3rd party plugins to override the template files.
	 *
	 * @param string $slug
	 * @param string $name
	 * @param array $variables
	 * @param boolean $echo
	 */
	function nevo_get_template_part( $slug, $name = null, $variables = array(), $echo = true ) {

		// Native WordPress action.
		do_action( 'get_template_part_' . $slug, $slug, $name );

		$templates = array();
		if ( isset( $name ) ) {
			$templates[] = $slug . '-' . $name . '.php';
		}

		$templates[] = $slug . '.php';

		// Native WordPress action.
		do_action( 'get_template_part', $slug, $name, $templates );

		// Get the final template file path.
		$template_file_path = false;

		// Iterate through the templates.
		foreach ( $templates as $template ) {
			
			// Check the template file in Child Theme.
			if ( file_exists( get_stylesheet_directory() . '/template-parts/' . $template ) ) {
				$template_file_path = get_stylesheet_directory() . '/template-parts/' . $template;
				break;
			}

			// Allow themes or plugins to add their own paths.
			$custom_paths = apply_filters( 'nevo_template_dirs', array() );

			// Sort the custom paths by key number.
			// Lower key number = higher priority.
			ksort( $custom_paths );

			// Iterate through the custom paths and use the path if it exists.
			foreach ( $custom_paths as $custom_path ) {
				$temp = trailingslashit( $custom_path ) . $template;

				if ( file_exists( $temp ) ) {
					$template_file_path = $temp;
					break 2; // break from 2 iteration levels, which are the $custom path and $templates.
				}
			}

			// Check the template file in Parent Theme.
			if ( file_exists( get_template_directory() . '/template-parts/' . $template ) ) {
				$template_file_path = get_template_directory() . '/template-parts/' . $template;
				break;
			}

			// Last resort, check the template file in WordPress theme compatibility files (very unlikely to reach here).
			elseif ( file_exists( ABSPATH . WPINC . '/theme-compat/' . $template ) ) {
				$template_file_path = ABSPATH . WPINC . '/theme-compat/' . $template;
				break;
			}
		}

		// Abort if no valid template file found.
		if ( empty( $template_file_path ) ) {
			return;
		}
		// Pass custom variables to the template file.

		foreach ( (array) $variables as $key => $value ) {
			set_query_var( $key, $value );
		}
		// Get the template part.
		ob_start();
		load_template( $template_file_path, false );
		$html = ob_get_clean();

		// Allow filters to modify the HTML markup.
		$html = apply_filters( 'nevo/template_part/' . $slug . ( ! empty( $name ) ? '-' . $name : '' ), $html );

		// Return or print the template part.
		if ( $echo ) {
			echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			return $html;
		}
	}
}

/**
 * Get page setting for various pages
 *
 * @param string $key Setting key.
 * @param mixed  $default Default value if setting key not found.
 * @return mixed Setting value.
 */
function nevo_get_page_setting($key, $default = null) {
    // Return null if no key specified.
    if (empty($key)) {
        return null;
    }

    // Apply filters and get initial value.
    $value = apply_filters('nevo_get_page_setting', null);

    switch ($key) {
        case is_search() && !is_archive():
            $value = Nevo()->get_setting('search_' . $key);
            break;
        case is_404():
            $value = Nevo()->get_setting('404_' . $key);
            break;
        case is_home() || is_date() || is_author():
			$individual_settings = wp_parse_args( get_term_meta( get_queried_object_id(), 'nevo_page_settings', true ), array() );
            $value = isset($individual_settings[$key]) ? $individual_settings[$key] : Nevo()->get_setting('post_archive_' . $key);
            break;
        case is_post_type_archive():
            $obj = get_queried_object();
            if ($obj) {
                $value = Nevo()->get_setting($obj->name . '_archive_' . $key);
            } else {
				$value = null;
			}
            break;
        case is_tax() || is_category() || is_tag():
			$obj = get_queried_object();
            if ($obj) {
                global $wp_taxonomies;
                $post_types = $wp_taxonomies[$obj->taxonomy]->object_type;
                $post_type = $post_types[0];
                $individual_settings = wp_parse_args(get_term_meta($obj->term_id, 'nevo_page_settings', true), array());
                $value = isset($individual_settings[$key]) ? $individual_settings[$key] : Nevo()->get_setting($post_type . '_archive_' . $key);
            } else {
				$value = null;
			}
            break;
        case is_singular():
            $obj = get_queried_object();
            if ($obj) {
                $individual_settings = wp_parse_args(get_post_meta($obj->ID, '_nevo_page_settings', true), array());
                if (!$individual_settings) {
                    if (Nevo()->is_woocommerce_active() && (is_cart() || is_checkout() || is_account_page())) {
                        $individual_settings = array(
                            'sidebar_layout' => 'no-sidebar',
                        );
                    }
                }
				// Use individual settings if option is specified.
				$value = isset($individual_settings[$key]) ? $individual_settings[$key] : Nevo()->get_setting( $obj->post_type . '_single_' . $key );
				
			} else {
				$value = null;
			}
            break;
        default:
            // If the value is empty, try to use the global value.
			if ( '' === $value || is_null( $value ) ) {
				$value = Nevo()->get_setting( $key, $default );
			}
            break;
    }
	// Allow developers to modify the value via filters.
	$value = apply_filters( 'nevo/page_settings/setting_value', $value, $key );
	$value = apply_filters( 'nevo/page_settings/setting_value/' . $key, $value );

	// Return the final value.
	return $value;
}

function nevo_get_post_setting($key, $default = null) {
    // Return null if no key specified.
    if (empty($key)) {
        return null;
    }

    // Apply filters and get initial value.
    $value = apply_filters('nevo_get_post_setting', null);
	if ( is_home() || is_date() || is_author() ) {
		$value = Nevo()->get_setting( 'post_archive_' . $key );
	}
	elseif ( is_category() || is_tag() ) {
		// Get settings on the individual term.
		$individual_settings = wp_parse_args( get_term_meta( get_queried_object_id(), 'nevo_page_settings', true ), array() );

		// Use individual settings if option is specified.
		if ( isset( $individual_settings[ $key ] ) ) {
			$value = $individual_settings[ $key ];
		}
		// Otherwise, use the Customizer value.
		else {
			$value = Nevo()->get_setting( 'post_archive_' . $key );
		}
	}
	elseif ( is_single() ) {
		$individual_settings = wp_parse_args( get_post_meta( get_the_ID(), '_nevo_page_settings', true ), array() );

		// Use individual settings if option is specified.
		if ( isset( $individual_settings[ $key ] ) ) {
			$value = $individual_settings[ $key ];
		}
		// Otherwise, use the Customizer value.
		else {
			$value = Nevo()->get_setting( 'post_single_' . $key );
		}
	}
	// Allow developers to modify the value via filters.
	$value = apply_filters( 'nevo/post_settings/setting_value', $value, $key );
	$value = apply_filters( 'nevo/post_settings/setting_value/' . $key, $value );
	
	// Return the final value.
	return $value;
}

/**
 * Explode list using "," and ", ". Giải mã danh sách bằng cách sử dụng "," và ",".
 *
 * @param string $string String to split up.
 * @return array Array of string parts.
 */
function nevo_list_explode( $string = '' ) {
	$string = str_replace( ', ', ',', $string );
	return explode( ',', $string );
}

/**
 * Call a shortcode function by tag name.
 *
 * @since  1.0
 *
 * @param string $tag     The shortcode whose function to call.
 * @param array  $atts    The attributes to pass to the shortcode function. Optional.
 * @param array  $content The shortcode's content. Default is null (none).
 *
 * @return string|bool False on failure, the result of the shortcode on success.
 */
function nevo_do_shortcode( $tag, array $atts = array(), $content = null ) {
	global $shortcode_tags;

	if ( ! isset( $shortcode_tags[ $tag ] ) ) {
		return false;
	}

	return call_user_func( $shortcode_tags[ $tag ], $atts, $content, $tag );
}

/**
 * Build our editor color palette based on our global colors.
 *
 * @since 1.0
 */
function nevo_get_editor_color_palette() {
	$global_colors = Nevo()->get_setting( 'global_color_style' );
	$editor_palette = array();
	$static_colors = false;

	if ( apply_filters( 'nevo_color_palette_use_static_colors', false ) ) {
		$static_colors = true;
	}
	if ( ! empty( $global_colors ) ) {
		foreach ( $global_colors['normal'] as $colorName => $colorValue ) {
			$editor_palette[] = array(
				'name'  => ucfirst($colorName) . ' color',//ucfirst(str_replace('_', ' ', $colorName)),
				'slug'  => $colorName,
				'color' => $static_colors ? $colorValue : 'var(--nvt-' . str_replace('_', '-', $colorName) . ')',
			);
		}
		// Add a new color to the color palette
		$editor_palette[] = array(
			'name'  => 'White Color',
			'slug'  => 'trang',
			'color' => 'var(--nvt-white)',
		);
	}
	return $editor_palette;
}

function nevo_build_category_tree( $taxonomies, $parent_id ) {
    $tree = array();
    foreach ( $taxonomies as $taxonomy ) {
        if ( $taxonomy->parent == $parent_id ) {
            $tree[ $taxonomy->term_id ]          = $taxonomy;
            $tree[ $taxonomy->term_id ]->children = nevo_build_category_tree( $taxonomies, $taxonomy->term_id );
        }
    }
    return $tree;
}

function nevo_display_category_tree( $category_tree, $selected_categories, $depth = 0 ) {
    foreach ( $category_tree as $category ) {
		$category_post_count = $category->count;
		$category_name = $category->name;
        ?>
        <option value="<?php echo esc_attr( $category->term_id ); ?>" <?php selected( in_array( $category->term_id, $selected_categories ) ); ?>>
            <?php echo str_repeat( '&nbsp;&nbsp;&nbsp;&nbsp;', $depth ); ?>
            <?php echo esc_html( "{$category_name} ({$category_post_count})" ); ?>
        </option>
        <?php
        if ( ! empty( $category->children ) ) {
            nevo_display_category_tree( $category->children, $selected_categories, $depth + 1 );
        }
    }
}

function nevo_get_dropdown_categories($taxonomy_name, $cat_value) {
    $taxonomies = get_terms(array(
        'taxonomy'   => $taxonomy_name,
        'orderby'    => 'name',
        'order'      => 'ASC',
        'hide_empty' => false
    ));

    if (!is_wp_error($taxonomies) && !empty($taxonomies)) {
        $category_tree = nevo_build_category_tree($taxonomies, 0);
        $choices = nevo_generate_choices($category_tree, $cat_value);
        return $choices;
    }

    return array();
}

function nevo_generate_choices($category_tree, $cat_value, $depth = 0) {
    $choices = array();
    $choices[''] = esc_html__('Select Category','nevo');
    foreach ($category_tree as $category) {
        $category_post_count = $category->count;
		$category_value  = $category->$cat_value;
        $category_name = "{$category->name} ({$category_post_count})";

        $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $depth);
        $value = esc_attr($category_value);
        $label = "{$indent}" . "{$category_name}";

		$choices[$value] = html_entity_decode($label);

        if (!empty($category->children)) {
            $child_choices = nevo_generate_choices($category->children, $cat_value, $depth + 1);
            $choices = array_merge($choices, $child_choices);
        }
    }
    return $choices;
}

if ( ! function_exists( 'get_data_columns' ) ) {
	/**
     * Get data columns with values.
     *
     * @access public
     * @param array $values
     * @return void
     */
    function get_data_columns( $values = [] ) {

        ob_start();

        if ( ! empty( $values ) ) {

            // Base or Mobile
            echo isset( $values['mobile'] )
                ? ' row-cols-' . esc_attr( $values['mobile'] ) .''
                : ( isset( $values['tablet'] )
                    ? ' row-cols-' . esc_attr( $values['tablet'] ) .''
                    : ( isset( $values['desktop'] )
                        ? ' row-cols-' . esc_attr( $values['desktop'] ) .''
                        : ''
                    )
                );
            // Tablet
            echo isset( $values['tablet'] ) && isset( $values['mobile'] )
                ? ' row-cols-md-' . esc_attr( $values['tablet'] ) .''
                : ( isset( $values['desktop'] ) && isset( $values['tablet'] )
                    ? ' row-cols-md-' . esc_attr( $values['desktop'] ) .''
                    : ''
                );
            // Desktop
            echo isset( $values['desktop'] ) && isset( $values['tablet'] ) && isset( $values['mobile'] )
                ? ' row-cols-lg-' . esc_attr( $values['desktop'] ) .''
                : '';
        }

        $output = ob_get_clean();

        echo apply_filters( 'nevo_get_data_columns', $output );
    }
}

if ( ! function_exists( 'nevo_create_visibility_classes' ) ) {
	function nevo_create_visibility_classes( $array, $prefix ) {
		if ( ! is_array( $array ) ) {
			return $prefix . '-' . $array;
		}
		$classes = array();
		$array   = array_reverse( $array );
		foreach ( $array as $k => $v ) {
			if ( 'desktop' == $k ) {
				$k = 'lg-';
			} elseif ( 'tablet' == $k ) {
				$k = 'sm-';
			} else {
				$k = '';
			}
			if ( 1 == $v ) {
				$v = 'block';
			} elseif ( 0 == $v ) {
				$v = 'none';
			}
			
			$classes[] = "{$prefix}-{$k}{$v}";
		}

		return join( ' ', $classes );
	}
}

if ( ! function_exists( 'nevo_create_visibility_elements' ) ) {
	function nevo_create_visibility_elements( $elements = array() ) {
		$hidden = false;
		foreach ( $elements as $element ) {
			$element = wp_parse_args(
				$element,
				array(
					'title'     => '',
					'elements'  => '',
					'_visibility' => '',
				)
			);
			if ( 'hidden' !== $element['_visibility'] ) {
				if ( 'title' === $element['elements'] ) {
					$hidden = true;
				}
			}
		}
		return $hidden;
	}
}

// Select page
function nevo_custom_select_field() {
    $page_args = array(
        'sort_order' => 'asc',
        'sort_column' => 'post_title',
        'hierarchical' => 1,
        'exclude' => '',
        'include' => '',
        'meta_key' => '',
        'meta_value' => '',
        'authors' => '',
        'parent' => -1,
        'exclude_tree' => '',
        'number' => '',
        'post_type' => 'page',
        'post_status' => 'publish'
    );
    $pages = get_pages($page_args);
    $page_options = array( '' => esc_html__( 'Select page', 'nevo' ) );
    foreach ( $pages as $page ) {
        $page_options[ $page->ID ] = $page->post_title;
    }
	return $page_options;
	
	/* $my_custom_select_array = array(
        'icon' => __( 'Icon', 'nevo' ),
        'image' => __( 'Image', 'nevo' ),
    );
    $my_custom_select_array += $page_options;
    return $my_custom_select_array; */
    ?>
    <select name="my_custom_select" id="my_custom_select">
        <?php foreach ( $page_options as $value => $label ) { ?>
            <option value="<?php echo $value; ?>"><?php echo $label; ?></option>
        <?php } ?>
    </select>
    <?php
}

function nevo_showon_frontpage() {
	return is_page_template('template-parts/frontpage.php');
}

if ( ! function_exists( 'nevo_load_section' ) ) {
	/**
	 * Load section
	 *
	 * @since 0.2
	 * @param $section_id
	 */
	function nevo_load_section( $section_id ) {
		/**
		 * Hook before section
		 */
		do_action( 'nevo_before_section_' . $section_id );
		if ( $section_id != 'hero' ) {
			do_action( 'nevo_before_section_part', $section_id );
		}

		nevo_get_template_part( 'section/section', $section_id );

		/**
		 * Hook after section
		 */
		if ( $section_id != 'hero' ) {
			do_action( 'nevo_after_section_part', $section_id );
		}
		do_action( 'nevo_after_section_' . $section_id );
	}
}

/**
 * Minify CSS string.
 * modified:
 * - add: rem to units
 * - add: remove space after (
 * - remove: remove space before (
 *
 * @param array $css
 * @return string
 */
function nevo_minify_css_string( $css ) {
	// Normalize whitespace
	$css = preg_replace( '/\s+/', ' ', $css );
	// Remove spaces before and after comment
	$css = preg_replace( '/(\s+)(\/\*(.*?)\*\/)(\s+)/', '$2', $css );
	// Remove comment blocks, everything between /* and */, unless
	// preserved with /*! ... */ or /** ... */
	$css = preg_replace( '~/\*(?![\!|\*])(.*?)\*/~', '', $css );
	// Remove ; before }
	$css = preg_replace( '/;(?=\s*})/', '', $css );
	// Remove space after , : ; { } ( */ >
	$css = preg_replace( '/(,|:|;|\{|}|\(|\*\/|>) /', '$1', $css );
	// Remove space before , ; { } ) >
	$css = preg_replace( '/ (,|;|\{|}|\)|>)/', '$1', $css );
	// Strips leading 0 on decimal values (converts 0.5px into .5px)
	$css = preg_replace( '/(:| )0\.([0-9]+)(%|rem|em|ex|px|in|cm|mm|pt|pc)/i', '${1}.${2}${3}', $css );
	// Strips units if value is 0 (converts 0px to 0)
	$css = preg_replace( '/(:| )(\.?)0(%|rem|em|ex|px|in|cm|mm|pt|pc)/i', '${1}0', $css );
	// Converts all zeros value into short-hand
	$css = preg_replace( '/0 0 0 0/', '0', $css );
	// Shortern 6-character hex color codes to 3-character where possible
	$css = preg_replace( '/#([a-f0-9])\\1([a-f0-9])\\2([a-f0-9])\\3/i', '#\1\2\3', $css );

	return trim( $css );
}

if ( ! function_exists( 'nevo_get_sidebar_layouts' ) ) {
	function nevo_get_sidebar_layouts() {
		return array(
			'default' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/default.png',
			),
			'content-narrow' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/no-sidebar-narrow.png',
			),
			'no-sidebar' 	=> array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-no-sidebar.png',
			),
			'left-sidebar' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-content.png',
			),
			'right-sidebar' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-sidebar.png',
			),
			'both-sidebars' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-content-sidebar.png',
			),
			'both-left' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-sidebar-content.png',
			),
			'both-right' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-sidebar-sidebar.png',
			),
		);
	}
}

if ( ! function_exists( 'nevo_get_post_archive_content_layouts' ) ) {
	function nevo_get_post_archive_content_layouts() {
		return apply_filters( 'nevo_get_post_archive_content_layouts', array(
			'content-1' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-1.png',
			),
			'content-2' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-2.png',
			),
			'content-3' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-3.png',
			),
			'content-4' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-4.png',
			),
			'content-5' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-5.png',
			),
			'content-6' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-6.png',
			),
			'content-7' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-7.png',
			),
			'content-8' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-8.png',
			),
			'content-25' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-25.png',
			),
			'content-9' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-9.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-10' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-10.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-11' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-11.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-12' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-12.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-13' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-13.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-14' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-14.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-15' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-15.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-16' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-16.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-17' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-17.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-18' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-18.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-19' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-19.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-20' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-20.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-21' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-21.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-22' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-22.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-23' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-23.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'content-24' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/content-24.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
		) );
	}
}

if ( ! function_exists( 'nevo_get_post_single_content_layouts' ) ) {
	function nevo_get_post_single_content_layouts() {
		return apply_filters( 'nevo_get_post_single_content_layouts', array(
			'single-post-1' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-1.png',
			),
			'single-post-2' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-2.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-3' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-3.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-4' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-4.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-5' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-5.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-6' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-6.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-7' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-7.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
			'single-post-8' => array(
				'img' => esc_url( get_template_directory_uri() ) . '/assets/images/content/single-post-8.png',
				'disable' => 1,
				'bubble'  => __( 'Pro', 'nevo' ),
			),
		) );
	}
}

add_action( 'wp_footer', 'nevo_do_a11y_scripts' );
/**
 * Enqueue scripts in the footer.
 *
 * @since 1.0
 */
function nevo_do_a11y_scripts() {
	if ( apply_filters( 'nevo_print_a11y_script', true ) ) {
		// Add our small a11y script inline.
		printf(
			'<script id="nevo-a11y">%s</script>',
			'!function(){"use strict";if("querySelector"in document&&"addEventListener"in window){var e=document.body;e.addEventListener("mousedown",function(){e.classList.add("using-mouse")}),e.addEventListener("keydown",function(){e.classList.remove("using-mouse")})}}();'
		);
	}
}
