<?php
/**
 * Navigation elements.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

add_filter( 'walker_nav_menu_start_el', 'nevo_walker_nav_menu_start_el', 99, 4 );
/**
 * Add dropdown caret to accordion menu item.
 *
 * @param string $item_output
 * @param WP_Post $item
 * @param integer $depth
 * @param stdClass $args
 * @return string
 */
function nevo_walker_nav_menu_start_el( $item_output, $item, $depth, $args ) {
	// Only add to menu item that has sub menu.
	if ( in_array( 'menu-item-has-children', $item->classes ) || in_array( 'page_item_has_children', $item->classes ) ) {
		// Only add to toggle menu.
		if ( isset($args->menu_class) && is_integer( strpos( $args->menu_class, 'menu-mobile' ) ) && $args->theme_location == 'mobile' ) {
			$sign = '<button class="nevo-toggle"><span class="nevo-icon">'. nevo_get_svg_icon( 'ui', 'bi-chevron-down', '', 20 ) .'</span><span class="screen-reader-text">' . esc_html__( 'Expand / Collapse', 'nevo' ) . '</span></button>';
			
			$item_output .= trim( $sign );
		}
	}

	return $item_output;
}

add_filter( 'wp_nav_menu_objects', 'nevo_custom_menu_class', 10, 2 );
function nevo_custom_menu_class( $items, $args ) {
    if( $args->theme_location == 'primary' || $args->theme_location == 'secondary' ) {
        $items[1]->classes[] = 'first';
		return $items;
    }
    return $items;
}

add_filter( 'nav_menu_item_title', 'nevo_nav_menu_item_title', 99, 4 );
/**
 * Add <span> wrapping tag and dropdown caret to menu item title.
 *
 * @param string $title
 * @param WP_Post $item
 * @param stdClass $args
 * @param integer $depth
 * @return string
 */
function nevo_nav_menu_item_title( $title, $item, $args, $depth ) {
	$sign = '';
	if( isset($args->menu_class)  && is_integer( strpos( $args->menu_class, 'nevo-nav-menu' ) ) && ( $args->theme_location == 'primary' || $args->theme_location == 'secondary' ) ) {
		// Only add to menu item that has sub menu.
		if ( in_array( 'menu-item-has-children', $item->classes ) || in_array( 'page_item_has_children', $item->classes ) ) {
			// Only add to hover menu.
			if ( is_rtl() ) {
				$sign = '<span class="nevo-icon cv-ms-1">'. nevo_get_svg_icon( 'ui', 0 < $depth ? 'bi-chevron-left' : 'bi-chevron-down', '', 16 ) .'</span>';
			} else {
				$sign = '<span class="nevo-icon cv-ms-1">'. nevo_get_svg_icon( 'ui', 0 < $depth ? 'bi-chevron-right' : 'bi-chevron-down', '', 16 ) .'</span>';
			}
		}
	}
	if( isset($args->menu_class) && is_integer( strpos( $args->menu_class, 'menu-vertical' ) ) && $args->theme_location == 'vertical' ) {
		// Only add to menu item that has sub menu.
		if ( in_array( 'menu-item-has-children', $item->classes ) || in_array( 'page_item_has_children', $item->classes ) ) {
			// Only add to hover menu.
			if ( is_rtl() ) {
				$sign = '<span class="nevo-icon vertical-icon-left">'. nevo_get_svg_icon( 'ui', 'bi-chevron-left', '', 16 ) .'</span>';
			} else {
				$sign = '<span class="nevo-icon vertical-icon-right">'. nevo_get_svg_icon( 'ui', 'bi-chevron-right', '', 16 ) .'</span>';
			}
		}
	}
	return '<span class="nevo-item-title" itemprop="name">' . $title . $sign . '</span>';
}

add_filter( 'nav_menu_link_attributes', 'nevo_nav_menu_link_attributes', 10, 4 );
/**
 * Add 'nevo-menu-link' class to menu item's anchor tag.
 *
 * @param array $atts
 * @param WP_Post $item
 * @param stdClass $args
 * @param int $depth
 */
function nevo_nav_menu_link_attributes( $atts, $item, $args, $depth ) {
	if ( ! isset( $atts['class'] ) ) {
		$atts['class'] 		= 'nevo-menu-link d-flex align-items-center';
		$atts['itemprop'] 	= 'url';
	}
	
	return $atts;
}

if ( ! function_exists( 'nevo_unassigned_menu' ) ) :
/**
 * Fallback HTML if there is no nav menu assigned to a navigation location.
 */
function nevo_unassigned_menu() {
	$labels = get_registered_nav_menus();

	if ( ! is_user_logged_in() || ! current_user_can( 'edit_theme_options' ) ) {
		return;
	}
	?>
	<ul class="nevo-nav-menu menu">
		<li class="menu-item">
			<a href="<?php echo esc_attr( add_query_arg( 'action', 'locations', admin_url( 'nav-menus.php' ) ) ); ?>" class="nevo-menu-link">
				<?php esc_html_e( 'Assign menu to this location', 'nevo' ); ?>
			</a>
		</li>
	</ul>
	<?php
}
endif;

if ( ! function_exists( 'nevo_menu_fallback' ) ) {
	/**
	 * Menu fallback.
	 *
	 * @since 1.0
	 *
	 * @param array $args Existing menu args.
	 */
	function nevo_menu_fallback( $args ) {
		
		?>
		<ul <?php nevo_do_element_classes( 'menu' ); ?>>
			<?php
			$args = array(
				'sort_column' => 'menu_order',
				'title_li' => '',
				'walker' => new Nevo_Page_Walker(),
			);

			wp_list_pages( $args );
			?>
		</ul>
		<?php
	}
}

if ( ! class_exists( 'Nevo_Page_Walker' ) && class_exists( 'Walker_Page' ) ) {
	/**
	 * Add current-menu-item to the current item if no theme location is set
	 * This means we don't have to duplicate CSS properties for current_page_item and current-menu-item
	 *
	 * @since 1.0
	 */
	class Nevo_Page_Walker extends Walker_Page {
		function start_el( &$output, $page, $depth = 0, $args = array(), $current_page = 0 ) { // phpcs:ignore
			$css_class = array( 'menu-item', 'page_item', 'page-item-' . $page->ID );
			$button = '';

			if ( isset( $args['pages_with_children'][ $page->ID ] ) ) {
				$css_class[] = 'menu-item-has-children';
				$icon = '<span class="nevo-icon cv-ms-1">'. nevo_get_svg_icon( 'ui', 'bi-chevron-down', '', 20 ) .'</span>';
				$button = $icon;
			}

			if ( ! empty( $current_page ) ) {
				$_current_page = get_post( $current_page );
				if ( $_current_page && in_array( $page->ID, $_current_page->ancestors ) ) {
					$css_class[] = 'current-menu-ancestor';
				}

				if ( $page->ID == $current_page ) { // phpcs:ignore
					$css_class[] = 'current-menu-item';
				} elseif ( $_current_page && $page->ID == $_current_page->post_parent ) { // phpcs:ignore
					$css_class[] = 'current-menu-parent';
				}
			} elseif ( $page->ID == get_option( 'page_for_posts' ) ) { // phpcs:ignore
				$css_class[] = 'current-menu-parent';
			}

			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core filter name.
			$css_classes = implode( ' ', apply_filters( 'page_css_class', $css_class, $page, $depth, $args, $current_page ) );

			$args['link_before'] = empty( $args['link_before'] ) ? '' : $args['link_before'];
			$args['link_after'] = empty( $args['link_after'] ) ? '' : $args['link_after'];

			$output .= sprintf(
				'<li class="%s"><a class="nevo-menu-link" href="%s">%s<span class="nevo-item-title" itemprop="name">%s</span>%s%s</a>',
				$css_classes,
				get_permalink( $page->ID ),
				$args['link_before'],
				apply_filters( 'the_title', $page->post_title, $page->ID ), // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core filter name.
				$args['link_after'],
				$button
			);
		}
	}
}