<?php
/**
 * Header elements.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'nevo_pingback_header' ) ) {
	add_action( 'wp_head', 'nevo_pingback_header' );
	/**
	 * Add a pingback url auto-discovery header for singularly identifiable articles.
	 *
	 * @since 1.0
	 */
	function nevo_pingback_header() {
		if ( is_singular() && pings_open() ) {
			printf( '<link rel="pingback" href="%s">' . "\n", esc_url( get_bloginfo( 'pingback_url' ) ) );
		}
	}
}

if ( ! function_exists( 'nevo_add_viewport' ) ) {
	add_action( 'wp_head', 'nevo_add_viewport', 1 );
	/**
	 * Add viewport to wp_head.
	 *
	 * @since 1.0
	 */
	function nevo_add_viewport() {
		echo apply_filters( 'nevo_meta_viewport', '<meta name="viewport" content="width=device-width, initial-scale=1">' );  // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

add_action( 'nevo_before_header', 'nevo_do_skip_to_content_link', 2 );
/**
 * Add skip to content link before the header.
 *
 * @since 1.0
 */
function nevo_do_skip_to_content_link() {
	printf(
		'<a class="screen-reader-text skip-link" href="#content" title="%1$s">%2$s</a>',
		esc_attr__( 'Skip to content', 'nevo' ),
		esc_html__( 'Skip to content', 'nevo' )
	);
}

if ( ! function_exists( 'nevo_construct_header' ) ) {
	add_action( 'nevo_header', 'nevo_construct_header' );
	/**
	 * Build the header.
	 *
	 * @since 1.0
	 */
	function nevo_construct_header() {
		?>
		<header <?php nevo_do_attr( 'header' ); ?>>
			<?php
			/**
			 * nevo_before_header_content hook.
			 *
			 * @since 0.1
			 */
			do_action( 'nevo_before_header_content' );

			/**
			 * nevo_header_content hook.
			 *
			 * @since 0.1
			 */
			do_action( 'nevo_header_content' );

			/**
			 * nevo_after_header_content hook.
			 *
			 * @since 0.1
			 */
			do_action( 'nevo_after_header_content' );
			?>
		</header>
		<?php
	}
}

if ( ! function_exists( 'nevo_construct_logo' ) ) {
	/**
	 * Build the logo
	 *
	 * @since 0.1
	 */
	function nevo_construct_logo() {
		$logo_url    	= get_theme_mod( 'custom_logo' );
		$logo_url    	= Nevo()->get_media( $logo_url, 'full' );
		$logo_url 		= esc_url( apply_filters( 'nevo_logo', $logo_url ) );
		$logo_retina  	= Nevo()->get_setting( 'header_logo_retina' );
		$retina_logo_url = esc_url( apply_filters( 'nevo_retina_logo', Nevo()->get_media( $logo_retina ) ) );

		// If we don't have a logo, bail.
		if ( empty( $logo_url ) ) {
			return;
		}

		/**
		 * nevo_before_logo hook.
		 *
		 * @since 0.1
		 */
		do_action( 'nevo_before_logo' );

		$attr = apply_filters(
			'nevo_logo_attributes',
			array(
				'class' => 'header-image is-logo-image',
				'alt'   => esc_attr( apply_filters( 'nevo_logo_title', get_bloginfo( 'name', 'display' ) ) ),
				'src'   => $logo_url,
				'title' => esc_attr( apply_filters( 'nevo_logo_title', get_bloginfo( 'name', 'display' ) ) ),
			)
		);

		if ( '' !== $retina_logo_url ) {
			$attr['srcset'] = $logo_url . ' 1x, ' . $retina_logo_url . ' 2x';

			// Add dimensions to image if retina is set. This fixes a container width bug in Firefox.
			if ( get_theme_mod( 'custom_logo' ) ) {
				$data = wp_get_attachment_metadata( get_theme_mod( 'custom_logo' ) );

				if ( ! empty( $data ) ) {
					$attr['width'] = $data['width'];
					$attr['height'] = $data['height'];
				}
			}
		}

		$attr = array_map( 'esc_attr', $attr );

		$html_attr = '';
		foreach ( $attr as $name => $value ) {
			$html_attr .= " $name=" . '"' . $value . '"';
		}
		$before_img = apply_filters( 'nevo_before_logo_img', null );
		$after_img = apply_filters( 'nevo_after_logo_img', null );
		
		$attrs_array = apply_filters( 'nevo_header_logo_attrs', array() );
		$attrs = '';
		foreach ( $attrs_array as $key => $value ) {
			$attrs .= ' ' . $key . '="' . esc_attr( $value ) . '"';
		}
		// Print our HTML.
		echo apply_filters( // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			'nevo_logo_output',
			sprintf(
				'<div class="site-logo" %6$s>
					<a href="%1$s" title="%2$s" rel="home">
						%4$s<img %3$s />%5$s
					</a>
				</div>',
				esc_url( apply_filters( 'nevo_logo_href', home_url( '/' ) ) ),
				esc_attr( apply_filters( 'nevo_logo_title', get_bloginfo( 'name', 'display' ) ) ),
				$html_attr,
				$before_img,
				$after_img,
				$attrs
			),
			$logo_url,
			$html_attr
		);

		/**
		 * nevo_after_logo hook.
		 *
		 * @since 0.1
		 */
		do_action( 'nevo_after_logo' );
	}
}