<?php
/**
 * Nevo WooCommerce Class
 *
 * @package  nevo
 * @since    1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Nevo_WooCommerce' ) ) :

	/**
	 * The Nevo WooCommerce Integration class
	 */
	class Nevo_WooCommerce {
		/**
		 * Singleton instance
		 *
		 * @var Nevo_WooCommerce
		 */
		static $instance;

		/**
		 * Get singleton instance.
		 *
		 * @return Nevo_WooCommerce
		 */
		static function instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Setup class.
		 *
		 * @since 1.0
		 */
		public function __construct() {
			add_action( 'after_setup_theme', array( $this, 'setup' ) );
			add_filter( 'body_class', array( $this, 'woocommerce_body_class' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'woocommerce_scripts' ), 20 );
			
			// Template hooks
			add_action( 'widgets_init', array( $this, 'register_sidebars' ) );
			add_action( 'init', array( $this, 'modify_template_hooks' ) );
			add_action( 'wp', array( $this, 'modify_template_hooks_after_init' ) );
			
			// Instead of loading Core CSS files, we only register the font families.
			add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );
			add_filter( 'wp_enqueue_scripts', array( $this, 'add_core_fonts' ), 130 );
			
			// Legacy WooCommerce columns filter.
			if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '3.3', '<' ) ) {
				add_action( 'woocommerce_before_shop_loop', 'nevo_product_columns_wrapper', 40 );
				add_action( 'woocommerce_after_shop_loop', 'nevo_product_columns_wrapper_close', 40 );
			}
			add_filter( 'woocommerce_show_page_title', '__return_null' );
			remove_action( 'woocommerce_archive_description', 'woocommerce_product_archive_description', 10 );
			remove_action( 'woocommerce_archive_description', 'woocommerce_taxonomy_archive_description', 10 );
			// Remove default WooCommerce breadcrumb.
			remove_action( 'woocommerce_before_main_content', 'woocommerce_breadcrumb', 20 );
			// Remove default WooCommerce wrappers.
			remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
			remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10 );
			// Change main content (primary) wrapper.
			add_action( 'woocommerce_before_main_content', 'nevo_output_content_wrapper_start', 10 );
			add_action( 'woocommerce_after_main_content', 'nevo_output_content_wrapper_end', 10 );
			// Change sidebar content.
			remove_action( 'woocommerce_sidebar', 'woocommerce_get_sidebar', 10 );
			add_action( 'woocommerce_sidebar', 'nevo_construct_sidebars', 10 );
			
			if ( is_admin() || is_customize_preview() ) {
				add_filter( 'Nevo_Control_Args', array( $this, 'add_product_url' ), 35 );
			}
			// Overwrite Categories Walker.
			add_filter( 'woocommerce_product_categories_widget_args', array( $this, 'nevo_wc_cat_list_args' ) );
			
			require_once get_template_directory() . '/inc/woocommerce/class-nevo-woocommerce-adjacent-products.php';
			require_once get_template_directory() . '/inc/woocommerce/nevo-woocommerce-template-hooks.php';
			require_once get_template_directory() . '/inc/woocommerce/nevo-woocommerce-template-functions.php';
			require_once get_template_directory() . '/inc/woocommerce/nevo-woocommerce-functions.php';
			require_once get_template_directory() . '/inc/woocommerce/configs/header-cart.php';
			require_once get_template_directory() . '/inc/woocommerce/configs/cart.php';
			require_once get_template_directory() . '/inc/woocommerce/configs/archive.php';
			require_once get_template_directory() . '/inc/woocommerce/configs/single.php';
		}
		
		/**
		 * Overwrite Categories Walker
		 *
		 * @see WC_Product_Cat_List_Walker
		 *
		 * @param array $args List args.
		 *
		 * @return mixed
		 */
		function nevo_wc_cat_list_args( $args ) {
			require_once get_template_directory() . '/inc/woocommerce/class-wc-product-cat-list-walker.php';
			$args['walker'] = new Nevo_WC_Product_Cat_List_Walker();

			return $args;
		}
		
		/**
		 * Sets up theme defaults and registers support for various WooCommerce features.
		 *
		 * Note that this function is hooked into the after_setup_theme hook, which
		 * runs before the init hook. The init hook is too late for some features, such
		 * as indicating support for post thumbnails.
		 *
		 * @since 1.0
		 * @return void
		 */
		public function setup() {
			add_theme_support(
				'woocommerce',
				apply_filters(
					'nevo_woocommerce_args',
					array(
						//'single_image_width'    => 555,
						//'thumbnail_image_width' => 324,
						'product_grid'          => array(
							'default_columns' => 3,
							'default_rows'    => 4,
							'min_columns'     => 1,
							'max_columns'     => 6,
							'min_rows'        => 1,
						),
					)
				)
			);

			add_theme_support( 'wc-product-gallery-zoom' );
			add_theme_support( 'wc-product-gallery-lightbox' );
			add_theme_support( 'wc-product-gallery-slider' );

			/**
			 * Add 'nevo_woocommerce_setup' action.
			 *
			 * @since  1.0
			 */
			do_action( 'nevo_woocommerce_setup' );
		}
		
		/**
		 * Register additional sidebar for WooCommerce.
		 */
		public function register_sidebars() {
			register_sidebar(
				array(
					'name'          => esc_html__( 'Right Sidebar Shop', 'nevo' ),
					'id'            => 'shop-1',
					'before_widget' => '<aside id="%1$s" class="widget inner-padding %2$s">',
					'after_widget'  => '</aside>',
					'before_title'  => apply_filters( 'nevo_start_widget_title', '<div class="heading-title"><h2 class="widget-title general-title">' ),
					'after_title'   => apply_filters( 'nevo_end_widget_title', '</h2></div>' ),
				)
			);
			register_sidebar(
				array(
					'name'          => esc_html__( 'Left Sidebar Shop', 'nevo' ),
					'id'            => 'shop-2',
					'before_widget' => '<aside id="%1$s" class="widget inner-padding %2$s">',
					'after_widget'  => '</aside>',
					'before_title'  => apply_filters( 'nevo_start_widget_title', '<div class="heading-title"><h2 class="widget-title general-title">' ),
					'after_title'   => apply_filters( 'nevo_end_widget_title', '</h2></div>' ),
				)
			);
		}
		
		/**
		 * Modify filters for WooCommerce template rendering.
		 */
		public function modify_template_hooks() {
			// Page settings
			add_filter( 'nevo/page_settings/setting_value', array( $this, 'modify_shop_page_settings' ), 10, 2 );
			add_filter( 'nevo_get_page_header_elements', array( $this, 'modify_page_header_elements' ) );
			add_filter( 'nevo_get_page_header_background_image', array( $this, 'modify_page_header_background_image' ) );
			
			// Whether to use theme's breadcrumb or WooCommerce original breadcrumb.
			add_filter( 'nevo_breadcrumb_html', array( $this, 'modify_breadcrumb_html' ) );
			
			// Change "Products" in theme's breadcrumb trails to Shop page's title.
			add_filter( 'nevo_breadcrumb_trail_items', array( $this, 'modify_theme_breadcrumb_trails' ) );
			
			add_filter( 'woocommerce_output_related_products_args', array( $this, 'related_products_args' ) );
			add_filter( 'woocommerce_product_thumbnails_columns', array( $this, 'thumbnail_columns' ) );
			add_filter( 'woocommerce_breadcrumb_defaults', array( $this, 'change_breadcrumb_delimiter' ) );
			add_filter( 'get_product_search_form', 'nevo_custom_product_search_form' );
			
			// Change default sidebar .
			add_filter( 'nevo_default_sidebar_right', array( $this, 'modify_theme_default_sidebar_right' ) );
			add_filter( 'nevo_default_sidebar_left', array( $this, 'modify_theme_default_sidebar_left' ) );
			
		}
		
		/**
		 * WP action
		 */
		public function modify_template_hooks_after_init() {
			if ( ! Nevo()->get_setting( 'product_archive_result_count' ) ) {
				remove_action( 'woocommerce_before_shop_loop', 'woocommerce_result_count', 20 );
			}
			if ( ! Nevo()->get_setting( 'product_archive_filtering' ) ) {
				remove_action( 'woocommerce_before_shop_loop', 'woocommerce_catalog_ordering', 30 );
			}
			// Change default woocommerce pagination .
			if ( is_woocommerce() ) {
				remove_action( 'woocommerce_after_shop_loop', 'woocommerce_pagination', 10 );
				remove_action( 'nevo_after_endwhile', 'nevo_do_post_navigation', 10 );
				add_action( 'woocommerce_after_shop_loop', 'nevo_do_post_navigation', 10 );
			}
			if ( ! Nevo()->get_setting( 'product_cart_cross_sell' ) ) {
				remove_action( 'woocommerce_cart_collaterals', 'woocommerce_cross_sell_display' );
			}
			if ( ! Nevo()->get_setting( 'product_single_tabs_description' ) ) {
				add_filter( 'woocommerce_product_description_heading', '__return_false', 999 );
			}
			if ( ! Nevo()->get_setting( 'product_single_tabs_information' ) ) {
				add_filter( 'woocommerce_product_additional_information_heading', '__return_false', 999 );
			}
			// Related product.
			if ( ! Nevo()->get_setting( 'product_single_related' ) ) {
				remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_related_products', 20 );
			}
			// Upsell product.
			if ( ! Nevo()->get_setting( 'product_single_upsells' ) ) {
				remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_upsell_display', 15 );
			}
			if ( is_product() && $page = nevo_create_visibility_elements( Nevo()->get_setting( 'product_single_page_header_elements' ) ) ) {
				remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_title', 5 );
			}
		}
		
		/**
		 * Modify Shop Page Settings.
		 *
		 * @param string $value, $key
		 * @return string
		 */
		public function modify_shop_page_settings( $value, $key ) {
			
			$page_id = wc_get_page_id( 'shop' );
			$individual_settings = wp_parse_args( get_post_meta( $page_id, '_nevo_page_settings', true ), array() );
			// Use individual settings if option is specified.
			if ( isset( $individual_settings[ $key ] ) && is_shop() ) {
				$value = $individual_settings[ $key ];
			}
			return $value;
		}
		
		/**
		 * Modify Page header.
		 *
		 * @param string $elements
		 * @return string
		 */
		public function modify_page_header_elements( $elements ) {
			if ( is_shop() || is_product_category() || is_product_tag() ) {
				$elements = Nevo()->get_setting( 'product_archive_page_header_elements' );
			}
			if ( is_product() ) {
				$elements = Nevo()->get_setting( 'product_single_page_header_elements' );
			}
			return $elements;
		}
		
		/**
		 * Modify Page header.
		 *
		 * @param string $bg_image
		 * @return string
		 */
		public function modify_page_header_background_image( $bg_image ) {
			$image = '';
			if ( is_shop() ) {
				$page_id  = wc_get_page_id( 'shop' );
				$image = Nevo()->get_setting( 'product_archive_page_header_background_image' );
				if ( $image === 'thumbnail' ) {
					$bg_image = get_the_post_thumbnail_url( $page_id, 'full' );
				}
			}
			if ( is_product_taxonomy() ) {
				global $wp_query;
				$cat          = $wp_query->get_queried_object();
				$thumbnail_id = get_term_meta( $cat->term_id, 'thumbnail_id', true );
				$image 	  = Nevo()->get_setting( 'product_archive_page_header_background_image' );
				
				if ( $image === 'thumbnail' ) {
					$bg_image = Nevo()->get_media( $thumbnail_id, 'full' );
				}
			}
			if ( is_product() ) {
				$image = Nevo()->get_setting( 'product_single_page_header_background_image' );
				if ( $image === 'thumbnail' ) {
					$bg_image = get_the_post_thumbnail_url( get_the_ID(), 'full' );
				}
			}
			return $bg_image;
		}
		
		/**
		 * Modify breadcrumb.
		 *
		 * @param string $html
		 * @return string
		 */
		public function modify_breadcrumb_html( $html ) {
			// Make sure it's WooCommerce page
			if ( is_woocommerce() ) {
				// If user chose not to use theme's breadcrumb, use WooCommerce breadcrumb.
				if ( apply_filters( 'woocommerce_breadcrumb_use_theme_module', false ) ) {
					ob_start();
					woocommerce_breadcrumb();
					$html = ob_get_clean();
				}
			}
			return $html;
		}
		
		/**
		 * Change "Products" in theme's breadcrumb trails to Shop page's title.
		 *
		 * @param array $items
		 * @return array
		 */
		public function modify_theme_breadcrumb_trails( $items ) {
			// Make sure it's WooCommerce page.
			if ( is_woocommerce() ) {
				// If there's archive page in the trail, change it.
				if ( isset( $items['post_type_archive'] ) ) {
					$items['post_type_archive']['label'] = get_the_title( wc_get_page_id( 'shop' ) );
				}

				// Build product categories trails on single product page.
				if ( is_product() ) {
					$terms = wc_get_product_terms(
						get_the_ID(),
						'product_cat',
						array(
							'orderby' => 'parent',
							'order'   => 'DESC',
						)
					);

					if ( $terms ) {
						$main_term = $terms[0];
						$parents = get_ancestors( $main_term->term_id, 'product_cat' );
		
						// Add parent categories.
						$i = count( $parents );
						while ( $i > 0 ) {
							$parent_term = get_term( $parents[ $i - 1 ], 'product_cat' );
							
							$cat_items['term_parent__' . $i ] = array(
								'label' => $parent_term->name,
								'url'   => get_term_link( $parent_term ),
							);
							
							$i--;
						}

						// Add direct main category.
						$cat_items['term'] = array(
							'label' => $main_term->name,
							'url'   => get_category_link( $main_term ),
						);
			
						// Insert the product categories into trails.
						$items = array_merge (
							array_slice( $items, 0, count( $items ) - 1 ),
							$cat_items,
							array_slice( $items, count( $items ) - 1, null )
						);
					}
				}
			}
			return $items;
		}

		/**
		 * Modify sidebar.
		 *
		 * @since 1.0
		 * @return void
		 */
		public function modify_theme_default_sidebar_right( $sidebar ) {
			// Make sure it's WooCommerce page
			if ( is_woocommerce() ) {
				$sidebar = 'shop-1';
			}
			return $sidebar;
		}
		public function modify_theme_default_sidebar_left( $sidebar ) {
			// Make sure it's WooCommerce page
			if ( is_woocommerce() ) {
				$sidebar = 'shop-2';
			}
			return $sidebar;
		}
		
		/**
		 * Add url to customize preview when section open
		 *
		 * @param array $args Args to add.
		 *
		 * @return mixed
		 */
		public function add_product_url( $args ) {

			$query = new WP_Query(
				array(
					'post_type'      => 'product',
					'posts_per_page' => 1,
					'orderby'        => 'rand',
				)
			);

			$products = $query->get_posts();
			if ( count( $products ) ) {
				$args['section_urls']['product_single'] = get_permalink( $products[0] );
			}
			$args['section_urls']['product_archive'] = get_permalink( wc_get_page_id( 'shop' ) );
			$args['section_urls']['product_cart'] = get_permalink( wc_get_page_id( 'cart' ) );
			$args['section_urls']['product_checkout'] = get_permalink( wc_get_page_id( 'checkout' ) );
			
			return $args;
		}
		
		/**
		 * Add CSS in <head> to register WooCommerce Core fonts.
		 *
		 * @since 1.0
		 * @return void
		 */
		public function add_core_fonts() {
			$fonts_url = plugins_url( '/woocommerce/assets/fonts/' );
			wp_add_inline_style(
				'nevo-woocommerce-style',
				'@font-face {
				font-family: star;
				src: url(' . $fonts_url . 'star.eot);
				src:
					url(' . $fonts_url . 'star.eot?#iefix) format("embedded-opentype"),
					url(' . $fonts_url . 'star.woff) format("woff"),
					url(' . $fonts_url . 'star.ttf) format("truetype"),
					url(' . $fonts_url . 'star.svg#star) format("svg");
				font-weight: 400;
				font-style: normal;
			}
			@font-face {
				font-family: WooCommerce;
				src: url(' . $fonts_url . 'WooCommerce.eot);
				src:
					url(' . $fonts_url . 'WooCommerce.eot?#iefix) format("embedded-opentype"),
					url(' . $fonts_url . 'WooCommerce.woff) format("woff"),
					url(' . $fonts_url . 'WooCommerce.ttf) format("truetype"),
					url(' . $fonts_url . 'WooCommerce.svg#WooCommerce) format("svg");
				font-weight: 400;
				font-style: normal;
			}'
			);
		}

		/**
		 * Add WooCommerce specific classes to the body tag
		 *
		 * @param  array $classes css classes applied to the body tag.
		 * @return array $classes modified to include 'woocommerce-active' class
		 */
		public function woocommerce_body_class( $classes ) {
			if ( is_tax( 'product_cat' ) ) {
				$cat = get_queried_object();
				if ( $cat->parent > 0  ) $classes[] = 'subcategory';
			}
			return $classes;
		}

		/**
		 * WooCommerce specific scripts & stylesheets
		 *
		 * @since 1.0.0
		 */
		public function woocommerce_scripts() {
			$suffix = Nevo()->get_asset_suffix();
			
			$css_file = get_template_directory_uri() . "/assets/css/woocommerce/woocommerce{$suffix}.css";
			if ( is_rtl() ) {
				$css_file = get_template_directory_uri() . "/assets/css/woocommerce/woocommerce-rtl{$suffix}.css";
			}
			
			wp_enqueue_style( 'nevo-woocommerce-style', $css_file, array(), Nevo::$version, );

			wp_enqueue_script( 'nevo-woocommerce', get_template_directory_uri() . "/assets/js/woocommerce/woocommerce{$suffix}.js", array(), Nevo::$version, true );
			//wp_enqueue_script( 'scroll-visible', get_template_directory_uri() . '/assets/js/woocommerce/scroll-visible.js', array(), Nevo::$version, true );

			/* wp_localize_script(
				'scroll-visible',
				'scrollvisible',
				apply_filters(
					'scroll_visible_js_args',
					array(
						'smooth' => true,
					)
				)
			); */
			
			//wp_register_script( 'nevo-header-cart', get_template_directory_uri() . '/assets/js/woocommerce/header-cart.js', array(), Nevo::$version, true );
			//wp_enqueue_script( 'nevo-header-cart' );

			if ( ! class_exists( 'Nevo_Sticky_Add_to_Cart' ) && is_product() ) {
				//wp_register_script( 'nevo-sticky-add-to-cart', get_template_directory_uri() . '/assets/js/woocommerce/sticky-add-to-cart' . $suffix . '.js', array(), Nevo::$version, true );
			}
			
		}

		/**
		 * Related Products Args
		 *
		 * @param  array $args related products args.
		 * @since 1.0.0
		 * @return  array $args related products args
		 */
		public function related_products_args( $args ) {
			$args = apply_filters(
				'nevo_related_products_args',
				array(
					'posts_per_page' => intval( Nevo()->get_setting( 'product_single_related_number' ) ),
					'columns'        => intval( Nevo()->get_setting( 'product_single_related_col' ) ),
				)
			);

			return $args;
		}

		/**
		 * Product gallery thumbnail columns
		 *
		 * @return integer number of columns
		 * @since  1.0.0
		 */
		public function thumbnail_columns() {
			$columns = 4;
			$layout = nevo_get_page_setting('sidebar_layout');
			
			if ( is_product() && $layout === 'no-sidebar' ) {
				$columns = 8;
			}
			// Get the current product object from the product ID
			$product = wc_get_product( get_the_ID() );
			
			// if $product is not an object, return 0
			if ( $product === false ) {
				return 0;
			}
			
			// Get the product gallery attachment IDs
			$attachment_ids = $product->get_gallery_image_ids();

			// if there are no product gallery images, return 0
			if ( ! $attachment_ids ) {
				return 0;
			}
			if ( is_product() && 4 > count( $attachment_ids ) ) {
				$columns = 4;//count( $attachment_ids )+1;
			}

			return intval( apply_filters( 'nevo_product_thumbnail_columns', $columns ) );
		}

		/**
		 * Query WooCommerce Extension Activation.
		 *
		 * @param string $extension Extension class name.
		 * @return boolean
		 */
		public function is_woocommerce_extension_activated( $extension = 'WC_Bookings' ) {
			return class_exists( $extension ) ? true : false;
		}

		/**
		 * Remove the breadcrumb delimiter
		 *
		 * @param  array $defaults The breadcrumb defaults.
		 * @return array           The breadcrumb defaults.
		 * @since 1.0
		 */
		public function change_breadcrumb_delimiter( $defaults ) {
			$separator = Nevo()->get_setting( 'global_breadcrumb_separator' );
			
			$defaults['delimiter']   = '<span class="separator">'. $separator .'</span>';
			$defaults['wrap_before'] = '<nav class="woocommerce-breadcrumb" aria-label="' . esc_attr__( 'breadcrumbs', 'nevo' ) . '">';
			$defaults['wrap_after']  = '</nav>';
			return $defaults;
		}

		/**
		 * Add "Includes" meta to parent cart items.
		 * Displayed only on handheld/mobile screens.
		 *
		 * @since  1.0
		 *
		 * @param  array $group_mode_data Group mode data.
		 * @return array
		 */
		public function bundles_group_mode_options_data( $group_mode_data ) {
			$group_mode_data['parent']['features'][] = 'parent_cart_item_meta';

			return $group_mode_data;
		}
	}

endif;

return new Nevo_WooCommerce();