<?php
/**
 * Nevo Individual Page Settings metabox
 *
 * @package Nevo
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Nevo_Metabox_Settings {
	/**
	 * Singleton instance
	 *
	 * @var Nevo_Metabox_Settings
	 */
	private static $instance;

	/**
	 * Get singleton instance.
	 *
	 * @return Nevo_Metabox_Settings
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	protected function __construct() {
		// Post meta box
		add_action( 'add_meta_boxes', array( $this, 'add_post_meta_box' ), 10, 2 );
		add_action( 'save_post', array( $this, 'save_post_meta_box' ) );

		// Term meta box
		add_action( 'admin_init', array( $this, 'init_term_meta_boxes' ) );

		// Render actions
		add_action( 'nevo/admin/metabox/page_settings/fields/page_header', array( $this, 'render_options__page_header_layout' ), 10 );
		add_action( 'nevo/admin/metabox/page_settings/fields/content', array( $this, 'render_options__content_layout' ), 10 );
		add_action( 'nevo/admin/metabox/page_settings/fields/disable', array( $this, 'render_options__disable' ), 10 );
	}

	/**
	 * Return page settings values of the specified object (post or term object).
	 *
	 * @param WP_Post|WP_Term $obj
	 * @return array
	 */
	public function get_values( $obj ) {
		$option_key = 'nevo_page_settings';

		if ( is_a( $obj, 'WP_Post' ) ) {
			$values = get_post_meta( $obj->ID, '_' . $option_key, true );
		} elseif ( is_a( $obj, 'WP_Term' ) ) {
			$values = get_term_meta( $obj->term_id, $option_key, true );
		} else {
			$values = array();
		}

		return $values;
	}
	
	/**
	 * Add page settings meta box to post edit page.
	 *
	 * @param string $post_type
	 * @param WP_Post $post
	 */
	public function add_post_meta_box( $post_type, $post ) {
		// Limit meta box to certain post types.
		$post_types = nevo_get_support_post_types();
		
		add_meta_box(
			'nevo_page_settings',
			/* translators: %s: theme name. */
			__( 'Nevo Page Settings ', 'nevo' ),
			array( $this, 'render_meta_box__post' ),
			$post_type,
			'side', // 'normal', 'side'
			apply_filters( 'nevo/admin/metabox/page_settings/priority', 'high' )
		);
	}

	/**
	 * Handle save action for page settings meta box on post edit page.
	 *
	 * @param integer $post_id
	 */
	public function save_post_meta_box( $post_id ) {
		// Check if our nonce is set.
		if ( ! isset( $_POST['nevo_post_page_settings_nonce'] ) ) return;
		
		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( sanitize_key( $_POST['nevo_post_page_settings_nonce'] ), 'nevo_post_page_settings' ) ) return;

		// If this is an autosave, our form has not been submitted, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;

		// Check the user's permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) ) return;

		// Sanitize values.
		$sanitized = array();

		if ( isset( $_POST['nevo_page_settings'] ) && is_array( $_POST['nevo_page_settings'] ) ) {
			$page_settings = array_map( 'sanitize_key', wp_unslash( $_POST['nevo_page_settings'] ) );

			foreach ( $page_settings as $key => $value ) {
				if ( '' === $value ) continue;

				// If value is 0 or 1, cast to integer.
				if ( '0' === $value || '1' === $value ) {
					$value = intval( $value );
				}

				$sanitized[ $key ] = $value;
			}
		}

		// Update the meta field in the database.
		update_post_meta( $post_id, '_nevo_page_settings', $sanitized );
	}

	/**
	 * Initialize meta box on all public taxonomies.
	 */
	public function init_term_meta_boxes() {
		$taxonomies = array_merge( apply_filters('nevo_init_term_meta_boxes', array( 'category', 'post_tag' ) ),
			get_taxonomies( array(
				'public'             => true,
				'publicly_queryable' => true,
				'rewrite'            => true,
				'_builtin'           => false,
			), 'names' )
		);
		foreach ( $taxonomies as $taxonomy ) {
			add_action( $taxonomy . '_add_form_fields', array( $this, 'render_meta_box__term_add' ) );
			add_action( $taxonomy . '_edit_form_fields', array( $this, 'render_meta_box__term_edit' ) );

			add_action( 'create_' . $taxonomy, array( $this, 'save_term_meta_box' ), 10, 2 );
			add_action( 'edit_' . $taxonomy, array( $this, 'save_term_meta_box' ), 10, 2 );
		}
	}

	/**
	 * Handle save action for page settings meta box on edit term page.
	 *
	 * @param integer $term_id
	 * @param integer $tt_id
	 */
	public function save_term_meta_box( $term_id, $tt_id ) {
		// Check if our nonce is set.
		if ( ! isset( $_POST['nevo_term_page_settings_nonce'] ) ) return;

		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( sanitize_key( $_POST['nevo_term_page_settings_nonce'] ), 'nevo_term_page_settings' ) ) return;

		// Sanitize values.
		$sanitized = array();
		if ( isset( $_POST['nevo_page_settings'] ) && is_array( $_POST['nevo_page_settings'] ) ) {
			$page_settings = array_map( 'sanitize_key', wp_unslash( $_POST['nevo_page_settings'] ) );
			
			foreach ( $page_settings as $key => $value ) {
				if ( '' === $value ) continue;

				// If value is 0 or 1, cast to integer.
				if ( '0' === $value || '1' === $value ) {
					$value = intval( $value );
				}
				
				$sanitized[ $key ] = $value;
			}
		}
		
		// Update the meta field in the database.
		update_term_meta( $term_id, 'nevo_page_settings', $sanitized );
	}

	/**
	 * Render page settings meta box on post / page edit page.
	 *
	 * @param WP_Post $post
	 */
	public function render_meta_box__post( $post ) {
		// Define an array of post IDs that will disable Individual Page Settings meta box.
		// The Individual Page Settings fields would not be displayed on those disabled IDs meta box.
		// Instead, The meta box would show the defined string specified on the disabled array.
		$disabled_ids = array();

		// Filter to modify disabled IDs.
		$disabled_ids = apply_filters( 'nevo/admin/metabox/page_settings/disabled_posts', $disabled_ids, $post );

		// Check if current post ID is one of the disabled IDs
		if ( array_key_exists( $post->ID, $disabled_ids ) ) {
			// Print the notice here.
			echo $disabled_ids[ $post->ID ]; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			// There is no other content should be rendered.
			return;
		}

		// Add a nonce field so we can check for it later.
		wp_nonce_field( 'nevo_post_page_settings', 'nevo_post_page_settings_nonce' );

		// Render meta box.
		$this->render_meta_box_content( $post );
	}

	/**
	 * Render page settings meta box on add term page.
	 */
	public function render_meta_box__term_add() {
		?>
		<div class="form-field nevo-add-term-page-settings" style="margin: 2em 0;">
			<h2>
				<?php
				/* translators: %s: theme name. */
				__( 'Nevo Page Settings ', 'nevo' );
				?>
			</h2>
			<?php
			// Add a nonce field so we can check for it later.
			wp_nonce_field( 'nevo_term_page_settings', 'nevo_term_page_settings_nonce' );

			// Render meta box
			echo '<div class="nevo-term-container">';
			$this->render_meta_box_content();
			echo '</div>';
			?>
		</div>
		<?php
	}

	/**
	 * Render page settings meta box on edit term page.
	 *
	 * @param string $taxonomy
	 */
	public function render_meta_box__term_edit( $term ) {
		?>
		<tr class="form-field nevo-edit-term-page-settings">
			<td colspan="2" style="padding: 0;">
				<h3>
					<?php
					/* translators: %s: tdeme name. */
					__( 'Nevo Page Settings ', 'nevo' );
					?>
				</h3>
				<?php
				// Add a nonce field so we can check for it later.
				wp_nonce_field( 'nevo_term_page_settings', 'nevo_term_page_settings_nonce' );
				
				// Render meta box
				echo '<div class="nevo-term-container">';
				$this->render_meta_box_content( $term );
				echo '</div>';
				?>
			</th>
		</tr>
		<?php
	}

	/**
	 * Render meta box wrapper.
	 *
	 * @param WP_Post|WP_Term $obj
	 */
	public function render_meta_box_content( $obj = null ) {
		nevo_get_admin_accor( $obj, 'metabox', apply_filters( 'nevo/admin/metabox/page_settings/tabs', array(
			'page_header'      => esc_html__( 'Page Header', 'nevo' ),
			'content'          => esc_html__( 'Content & Sidebar Layout', 'nevo' ),
			'disable'          => esc_html__( 'Disable Elements', 'nevo' ),
		) ) );
	}

	/**
	 * Render page settings meta box fields for Page Header.
	 *
	 * @param WP_Post|WP_Term $obj
	 */
	public function render_options__page_header_layout( $obj ) {
		$option_key = 'nevo_page_settings';
		$values = $this->get_values( $obj );
		
		$key = 'page_header_layout';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'radioimage',
			'label' 	  => esc_html__( 'Page Header Layout', 'nevo' ),
			'field_class' => 'no-image',
			'choices'     => array(
				'' => array(
					'field_label' => esc_html__( 'Customizer', 'nevo' ),
				),
				'nomal' => array(
					'field_label' => esc_html__( 'Contained', 'nevo' ),
				),
				'cv-mw-100' => array(
					'field_label' => esc_html__( 'Full Width', 'nevo' ),
				),
			),
			'value'       => nevo_array_value( $values, $key ),
		) );
		
		$description = get_post_type() === 'post' ? esc_html__( 'Choose position "Before Content" only applies to "Content Layout" 1 and 7.', 'nevo' ) : '';
		$key = 'page_header_display';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'select',
			'label' 	  => esc_html__( 'Page Header Display', 'nevo' ),
			'description' => $description,
			'choices'     => array(
				''    	  => esc_html__( 'Customizer', 'nevo' ),
				'disable' => esc_html__( 'Disable', 'nevo' ),
				'before'  => esc_html__( 'Before Content', 'nevo' ),
				'after'   => esc_html__( 'After Header', 'nevo' ),
			),
			'value'       => nevo_array_value( $values, $key ),
		) );
	}
	/**
	 * Render page settings meta box fields for Content & Sidebar layout.
	 *
	 * @param WP_Post|WP_Term $obj
	 */
	public function render_options__content_layout( $obj ) {
		$option_key = 'nevo_page_settings';
		$values = $this->get_values( $obj );
		
		$key = 'sidebar_layout';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'radioimage',
			'label' 	  => esc_html__( 'Sidebar Layout', 'nevo' ),
			'choices'     => array(
				'' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/custom.png',
				),
				'content-narrow' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/no-sidebar-narrow.png',
				),
				'no-sidebar' 	=> array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-no-sidebar.png',
				),
				'left-sidebar' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-content.png',
				),
				'right-sidebar' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-sidebar.png',
				),
				'both-sidebars' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-content-sidebar.png',
				),
				'both-left' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/sidebar-sidebar-content.png',
				),
				'both-right' => array(
					'img' => esc_url( get_template_directory_uri() ) . '/assets/images/sidebar/content-sidebar-sidebar.png',
				),
			),
			'value'       => nevo_array_value( $values, $key ),
		) );
		
		$key = 'content_container';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'radioimage',
			'label' 	  => esc_html__( 'Content Container', 'nevo' ),
			'field_class' => 'no-image',
			'choices'     => array(
				'' => array(
					'field_label' => esc_html__( 'Default', 'nevo' ),
				),
				'true' => array(
					'field_label' => esc_html__( 'Full Width', 'nevo' ),
				),
				'contained' => array(
					'field_label' => esc_html__( 'Contained', 'nevo' ),
				),
			),
			'value'       => nevo_array_value( $values, $key ),
		) );
	}

	/**
	 * Render page settings meta box fields for Elements.
	 *
	 * @param WP_Post|WP_Term $obj
	 */
	public function render_options__disable( $obj ) {
		$option_key = 'nevo_page_settings';
		$values = $this->get_values( $obj );
		
		$key = 'disable_header';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Header Desktop', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
		$key = 'disable_mobile_header';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Header Mobile/Tablet', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
		$key = 'disable_breadcrumb';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Breadcrumb', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
		
		if ( is_a( $obj, 'WP_Post' ) ) {
			$post_type = get_post_type( $obj );
			$key = 'disable_content_title';
			nevo_get_admin_field( array(
				'name'        => $option_key . '[' . $key . ']',
				'type'        => 'checkbox',
				'class'       => 'remove',
				'field_label' => esc_html__( 'Content Title', 'nevo' ),
				'value'       => nevo_array_value( $values, $key ),
			) );
		}
		
		$key = 'disable_footer_top';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Footer Top', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
		$key = 'disable_footer_main';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Footer Main', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
		$key = 'disable_footer_bottom';
		nevo_get_admin_field( array(
			'name'        => $option_key . '[' . $key . ']',
			'type'        => 'checkbox',
			'class'       => 'remove',
			'field_label' => esc_html__( 'Footer Bottom', 'nevo' ),
			'value'       => nevo_array_value( $values, $key ),
		) );
	}

}

Nevo_Metabox_Settings::instance();