<?php
/**
 * Nevo Theme Customizer Builder
 *
 * @package Nevo
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Header Builder and Customizer Options
 *
 */

class Nevo_Customizer_Header_Builder {

    /**
     * Panel ID, use for builder ID too
     *
     * @var string
     */
    public $panel = 'nevo_header';

    /**
     * Builder Sections and Controller ID
     *
     * @var string
     *
     */
    public $builder_section_controller = 'nevo_header_builder_controller_section';

    /**
     * Main Instance
     *
     * Insures that only one instance of Nevo_Customizer_Header_Builder exists in memory at any one
     * time. Also prevents needing to define globals all over the place.
     *
     * @return object
     */
    public static function instance() {

        // Store the instance locally to avoid private static replication
        static $instance = null;

        // Only run these methods if they haven't been ran previously
        if ( null === $instance ) {
            $instance = new Nevo_Customizer_Header_Builder;
        }

        // Always return the instance
        return $instance;
    }

    /**
     *  Run functionality with hooks
     *
     * @return void
     */
    public function run() {

        add_action( 'customize_register', array( $this, 'set_customizer' ), 1 );

        add_filter( 'nevo/customizer/config', array( $this, 'nevo_customizer_header_builder_config' ), 10 );

        add_filter( 'nevo_default_theme_options', array( $this, 'header_defaults' ) );
		
        add_filter( 'nevo_builders', array( $this, 'header_builder' ) );

        add_action( 'nevo_header_content', array( $this, 'nevo_header_display' ), 100 );
    }

    /**
     * Callback functions for customize_register,
     * Fixed previous array issue
     *
     * @param null
     * @return void
     */
    public function set_customizer() {
        $builder = nevo_get_header_builder_options( Nevo_Customizer_Header_Builder()->builder_section_controller );
        if ( is_array( $builder ) ) {
            $builder = json_encode( urldecode_deep( $builder ), true );
        }
        set_theme_mod( Nevo_Customizer_Header_Builder()->builder_section_controller, $builder );
    }

    /**
     * Get header builder
     *
     * @param null
     * @return void
     */
    public function get_builder() {
        $builder = nevo_get_header_builder_options( Nevo_Customizer_Header_Builder()->builder_section_controller );
        if ( ! is_array( $builder ) ) {
            $builder = json_decode( urldecode_deep( $builder ), true );
        }
        return $builder;
    }

    /**
     * Callback functions for nevo_default_theme_options,
     * Add Header Builder defaults values
     *
     * @param array $default_options
     * @return array
     */
    public function header_defaults( $default_options = array() ) {

        $header_defaults = [

            $this->builder_section_controller => [
                'desktop'   => [
                    'main'      => [
                        'nc-col-0'      => [
                            [
                                'id'    => 'title-tagline'
                            ]
                        ],
                        'nc-col-2'     => [
                            [
                                'id'    => 'primary-menu'
                            ],
							[
                                'id'    => 'account'
                            ]
                        ]
                    ]
                ],
                'mobile'    => [
                    'main'      => [
                        'nc-col-0'      => [
                            [
                                'id'    => 'title-tagline'
                            ]
                        ],
                        'nc-col-2'     => [
                            [
                                'id'    => 'toggle-menu'
                            ]
                        ]
                    ]
                ],
            ]
        ];
        return array_merge( $default_options, $header_defaults );
    }

    /**
     * Callback functions for nevo_builders,
     * Add Header Builder elements
     *
     * @param array $builder builder fields
     * @return array
     */
    public function header_builder( $builder ) {
        $items = apply_filters(
            'Nevo_Customizer_Header_Builder_items',
            array(
                'site_identity'   => array(
                    'name'    => esc_html__( 'Site Identity', 'nevo' ),
                    'id'      => 'title-tagline',
                    'section' => 'title_tagline',
                ),
                'primary_menu' => array(
                    'name'    => esc_html__( 'Primary Menu', 'nevo' ),
                    'id'      => 'primary-menu',
                    'section' => 'primary_menu',
                ),
                'toggle_menu'  => array(
                    'name'    => esc_html__( 'Toggle Menu', 'nevo' ),
                    'id'      => 'toggle-menu',
                    'section' => 'mobile_menu',
                ),
                'social_icons'    => array(
                    'name'    => esc_html__( 'Social Icons', 'nevo' ),
                    'id'      => 'social-icons',
                    'section' => 'social_icons',
                ),
                'search_icon'  => array(
                    'name'    => esc_html__( 'Search Icon', 'nevo' ),
                    'id'      => 'search-icon',
                    'section' => 'search_icon',
                ),
				'search_box'  => array(
                    'name'    => esc_html__( 'Search Box', 'nevo' ),
                    'id'      => 'search-box',
                    'section' => 'search_box',
                ),
                'button'   => array(
                    'name'    => esc_html__( 'Button', 'nevo' ),
                    'id'      => 'button',
                    'section' => 'header_button',
                ),
                'account'  => array(
                    'name'    => esc_html__( 'Account', 'nevo' ),
                    'id'      => 'account',
                    'section' => 'header_account',
                ),
                'html'  => array(
                    'name'    => esc_html__( 'HTML 1', 'nevo' ),
                    'id'      => 'html-1',
                    'section' => 'header_html',
                ),
            )
        );
		if ( Nevo()->is_woocommerce_active() ) {
			$items['cart']  = array(
				'name'    => esc_html__( 'Cart', 'nevo' ),
				'id'      => 'cart',
				'section' => 'header_cart',
			);
		}

        $header_builder = array(
            Nevo_Customizer_Header_Builder()->panel => array(
                'id'         => Nevo_Customizer_Header_Builder()->panel,
                'title'      => esc_html__( 'Header Builder', 'nevo' ),
                'control_id' => Nevo_Customizer_Header_Builder()->builder_section_controller,
                'panel'      => Nevo_Customizer_Header_Builder()->panel,
                'section'    => Nevo_Customizer_Header_Builder()->builder_section_controller,
                'devices'    => array(
                    'desktop' => esc_html__( 'Desktop', 'nevo' ),
                    'mobile'  => esc_html__( 'Mobile/Tablet', 'nevo' ),
                ),
                'items'      => $items,
                'rows'       => array(
                    'top'       => esc_html__( 'Header Top', 'nevo' ),
                    'main'      => esc_html__( 'Header Main', 'nevo' ),
                    'bottom'    => esc_html__( 'Header Bottom', 'nevo' ),
                ),
                'cols'       => array(
                    'top'       => 3,
                    'main'      => 3,
                    'bottom'    => 3,
                ),
				'builder'		=> 'header',
				'sidebar'		=> 'no-sidebar'
            ),
        );
        $header_builder = apply_filters( 'Nevo_Customizer_Header_Builder', $header_builder );
        return array_merge( $builder, $header_builder );

    }
	
	/**
     * Callback functions for customize_register,
     * Add Panel Section control
     *
     * @param object $wp_customize
     * @return void
     */
    public function nevo_customizer_header_builder_config( $configs = array() ) {
		$header_defaults = self::header_defaults();
		$config = array(
			array(
				'name'  => $this->panel,
				'type'  => 'panel',
				'priority' => 5,
				'title' => __( 'Header', 'nevo' ),
			),
			array(
				'name'  => $this->builder_section_controller,
				'type'  => 'section',
				'panel' => $this->panel,
				'priority' => 10,
				'title' => __( 'Header', 'nevo' ),
			),
			array(
				'name'            => $this->builder_section_controller,
				'type'            => 'text',
				'section'         => $this->builder_section_controller,
				'theme_supports'  => '',
				'selector'        => '.site-header',
				'render_callback' => 'nevo_construct_header',
				'sanitize_callback' => 'nevo_sanitize_field',
				'priority' 		  => 5,
				'title'           => __( 'Text', 'nevo' ),
				'default'         => $header_defaults,
			),
		);
		return array_merge( $configs, $config );
	}
	
	/**
     *Column Element
     *
     * @param $column_elements
     */
    public function column_elements( $column_elements) {
		foreach ( $column_elements as $element ) {
			$id     = esc_html($element['id']);
			nevo_header_element( $id );
		}
    }
	
    /**
     * Callback Function For nevo_action_header
     * Display Header Content
     *
     * @return void
     */
    public function nevo_header_display() {

        $builder = $this->get_builder();
        $active_sidebar = [];

        // Desktop Display
        if ( isset( $builder['desktop'] ) && ! empty( $builder['desktop'] ) ) {

            $desktop_builder_data   = [];
            $desktop_builder = $builder['desktop'];

            foreach ( $desktop_builder as $key => $single_row ) {

                if ( ! empty( $single_row ) ) {

                    foreach ( $single_row as $col_key => $columns ) {

                        if ( ! empty( $columns ) ) {

                            $desktop_builder_data[$key][$col_key] = $columns;
                        }

                    }

                }

            }
            if ( ! empty( $desktop_builder_data ) ) {
                $this->desktop_header( $desktop_builder_data );
            }

        }
        // Tablet/Mobile Display
        if ( isset( $builder['mobile'] ) && ! empty( $builder['mobile'] ) ) {

            $mobile_builder_data    = [];
            $mobile_builder = $builder['mobile'];

            foreach ( $mobile_builder as $key => $single_row ) {

                if ( ! empty( $single_row ) ) {

                    foreach ( $single_row as $col_key => $columns ) {

                        if ( ! empty( $columns ) ) {

                            $mobile_builder_data[$key][$col_key] = $columns;
                        }

                    }

                }

            }
            if ( ! empty( $mobile_builder_data ) ) {
                $this->mobile_header( $mobile_builder_data );
            }
        }
    }

    /**
     * Display Desktop Header Content
     *
     * @return void
     */
    public function desktop_header( $desktop_builder ) {
		if ( nevo_get_page_setting( 'disable_header' ) ) {
			return;
		}
        ?>
        <div id="nevo-desktop-header" <?php nevo_do_element_classes( 'desktop_header', array( 'desktop-header', 'd-none', 'd-lg-block' ) ); ?>>
			<?php
			if ( isset( $desktop_builder['top'] ) ) {
				$top_elements   = $desktop_builder['top'];
				
				$layout 	= Nevo()->get_setting( 'nevo_header_top_layout' );
				$highlight 	= Nevo()->get_setting( 'header_top_menu_highlight' );
				
				// Top Header Class
				$top_header_class = ['header-section top-header '. esc_attr( $highlight ) .''];
				
				if ( !empty( array_key_exists('nc-col-0', $top_elements ) ) ) {
					$top_header_class['start'] = 'has-start';
				}else{
					$top_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $top_elements ) ) ) {
					$top_header_class['center'] = 'has-center';
				}else{
					$top_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $top_elements ) ) ) {
					$top_header_class['end'] = 'has-end';
				}else{
					$top_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'header_top', array( ''. esc_attr( implode( ' ', $top_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">
						<div class="nevo-container inside-header <?php echo esc_attr( $layout ); ?>">
						
							<?php if ( array_key_exists( 'nc-col-0', $top_elements ) || array_key_exists('nc-col-2', $top_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $top_elements ) ) :
										$this->column_elements( $top_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $top_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $top_elements ) ) :
										$this->column_elements( $top_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $top_elements ) || array_key_exists('nc-col-2', $top_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $top_elements ) ) :
										$this->column_elements( $top_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</div><!-- .top-header -->
				<?php
			}
			if ( isset( $desktop_builder['main'] ) ) {
				$main_elements   = $desktop_builder['main'];
				
				$layout 	= Nevo()->get_setting( 'nevo_header_main_layout' );
				$highlight 	= Nevo()->get_setting( 'header_main_menu_highlight' );
				
				// Main Header Class
				$main_header_class = ['header-section main-header '. esc_attr( $highlight ) .''];
				
				if ( !empty( array_key_exists('nc-col-0', $main_elements ) ) ) {
					$main_header_class['start'] = 'has-start';
				}else{
					$main_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $main_elements ) ) ) {
					$main_header_class['center'] = 'has-center';
				}else{
					$main_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $main_elements ) ) ) {
					$main_header_class['end'] = 'has-end';
				}else{
					$main_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'header_main', array( ''. esc_attr( implode( ' ', $main_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">	
						<div class="nevo-container inside-header <?php echo esc_attr( $layout ); ?>">
						
							<?php if ( array_key_exists( 'nc-col-0', $main_elements ) || array_key_exists('nc-col-2', $main_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $main_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $main_elements ) || array_key_exists('nc-col-2', $main_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</div><!-- .main-header -->
				<?php
			}
			if ( isset( $desktop_builder['bottom'] ) ) {
				$bottom_elements   = $desktop_builder['bottom'];

				$layout 	= Nevo()->get_setting( 'nevo_header_bottom_layout' );
				$highlight 	= Nevo()->get_setting( 'header_bottom_menu_highlight' );
				
				// Bottom Header Class
				$bottom_header_class = ['header-section bottom-header '. esc_attr( $highlight ) .''];
				
				if ( !empty( array_key_exists('nc-col-0', $bottom_elements ) ) ) {
					$bottom_header_class['start'] = 'has-start';
				}else{
					$bottom_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $bottom_elements ) ) ) {
					$bottom_header_class['center'] = 'has-center';
				}else{
					$bottom_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $bottom_elements ) ) ) {
					$bottom_header_class['end'] = 'has-end';
				}else{
					$bottom_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'header_bottom', array( ''. esc_attr( implode( ' ', $bottom_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">	
						<div class="nevo-container inside-header <?php echo esc_attr( $layout ); ?>">
						
							<?php if ( array_key_exists( 'nc-col-0', $bottom_elements ) || array_key_exists('nc-col-2', $bottom_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $bottom_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $bottom_elements ) || array_key_exists('nc-col-2', $bottom_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</div><!-- .bottom-header -->
				<?php
			}
			?>
        </div><!-- #desktop-header -->
        <?php

    }

    /**
     * Display Mobile Header Content
     *
     * @return void
     */
    public function mobile_header( $mobile_builder ) {
        if ( nevo_get_page_setting( 'disable_mobile_header' ) ) {
			return;
		}
		?>
         <div id="nevo-mobile-header" <?php nevo_do_element_classes( 'mobile_header', array( 'mobile-header', 'd-block', 'd-lg-none' ) ); ?>>
			<?php
			if ( isset( $mobile_builder['top'] ) ) {
				$top_elements   = $mobile_builder['top'];

				// Top Header Class
				$top_header_class = ['header-section top-header'];
				
				if ( !empty( array_key_exists('nc-col-0', $top_elements ) ) ) {
					$top_header_class['start'] = 'has-start';
				}else{
					$top_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $top_elements ) ) ) {
					$top_header_class['center'] = 'has-center';
				}else{
					$top_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $top_elements ) ) ) {
					$top_header_class['end'] = 'has-end';
				}else{
					$top_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'mobile_header_top', array( ''. esc_attr( implode( ' ', $top_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">	
						<div class="nevo-container inside-header">
						
							<?php if ( array_key_exists( 'nc-col-0', $top_elements ) || array_key_exists('nc-col-2', $top_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $top_elements ) ) :
										$this->column_elements( $main_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $top_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $top_elements ) ) :
										$this->column_elements( $main_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $top_elements ) || array_key_exists('nc-col-2', $top_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $top_elements ) ) :
										$this->column_elements( $top_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
							
						</div>
					</div>
				</div><!-- .top-header -->
				<?php
			}
			if ( isset( $mobile_builder['main'] ) ) {
				$main_elements   = $mobile_builder['main'];

				// Main Header Class
				$main_header_class = ['header-section main-header'];
				
				if ( !empty( array_key_exists('nc-col-0', $main_elements ) ) ) {
					$main_header_class['start'] = 'has-start';
				}else{
					$main_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $main_elements ) ) ) {
					$main_header_class['center'] = 'has-center';
				}else{
					$main_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $main_elements ) ) ) {
					$main_header_class['end'] = 'has-end';
				}else{
					$main_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'mobile_header_main', array( ''. esc_attr( implode( ' ', $main_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">
						<div class="nevo-container inside-header">
						
							<?php if ( array_key_exists( 'nc-col-0', $main_elements ) || array_key_exists('nc-col-2', $main_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $main_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $main_elements ) || array_key_exists('nc-col-2', $main_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $main_elements ) ) :
										$this->column_elements( $main_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
							
						</div>
					</div>
				</div><!-- .main-header -->
				<?php
			}
			if ( isset( $mobile_builder['bottom'] ) ) {
				$bottom_elements   = $mobile_builder['bottom'];

				// Bottom Header Class
				$bottom_header_class = ['header-section bottom-header'];
				
				if ( !empty( array_key_exists('nc-col-0', $bottom_elements ) ) ) {
					$bottom_header_class['start'] = 'has-start';
				}else{
					$bottom_header_class['start'] = 'start-empty';
				}
				if ( !empty( array_key_exists('nc-col-1', $bottom_elements ) ) ) {
					$bottom_header_class['center'] = 'has-center';
				}else{
					$bottom_header_class['center'] = 'center-empty';
				}
				if ( !empty( array_key_exists('nc-col-2', $bottom_elements ) ) ) {
					$bottom_header_class['end'] = 'has-end';
				}else{
					$bottom_header_class['end'] = 'end-empty';
				}
				?>
				<div <?php nevo_do_element_classes( 'mobile_header_bottom', array( ''. esc_attr( implode( ' ', $bottom_header_class ) ) .'' ) ); ?>>
					<div class="nevo-section-inner">
						<div class="nevo-container inside-header">
						
							<?php if ( array_key_exists( 'nc-col-0', $bottom_elements ) || array_key_exists('nc-col-2', $bottom_elements ) ) : ?>
								<div class="header-col col-start d-flex justify-content-start align-items-center text-start h-100">
									<?php
									if ( array_key_exists('nc-col-0', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-0'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-1', $bottom_elements ) ) : ?>
								<div class="header-col col-center d-flex justify-content-center align-items-center text-center h-100">
									<?php
									if ( array_key_exists('nc-col-1', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-1'] );
									endif;
									?>
								</div>
							<?php endif; ?>

							<?php if ( array_key_exists( 'nc-col-0', $bottom_elements ) || array_key_exists('nc-col-2', $bottom_elements ) ) : ?>
								<div class="header-col col-end d-flex justify-content-end align-items-center text-end h-100">
									<?php
									if ( array_key_exists('nc-col-2', $bottom_elements ) ) :
										$this->column_elements( $bottom_elements['nc-col-2'] );
									endif;
									?>
								</div>
							<?php endif; ?>
							
						</div>
					</div>
				</div><!-- .bottom-header -->
				<?php
			}
			?>
        </div><!-- #mobile-header -->
        <?php
    }

}

/**
 * Create Instance for Nevo_Customizer_Header_Builder
 *
 * @param
 * @return object
 */
if ( ! function_exists( 'nevo_customizer_header_builder' ) ) {

    function nevo_customizer_header_builder() {

        return Nevo_Customizer_Header_Builder::instance();
    }

    nevo_customizer_header_builder()->run();
}

/**
 * Get header builder default options
 *
 * @param null
 * @return mixed nevo_theme_options
 *
 */
if ( ! function_exists( 'nevo_get_header_builder_options' ) ) :
    function nevo_get_header_builder_options( $key = '' ) {
        if ( ! empty( $key ) ) {
            $header_default_values  = Nevo_Customizer_Header_Builder()->header_defaults();
            $theme_mod_values       = get_theme_mod( $key, isset( $header_default_values[ $key ] ) ? $header_default_values[ $key ] : '' );
            return apply_filters( 'nevo_' . $key, $theme_mod_values );
        }
        return false;
    }
endif;

if ( ! function_exists( 'nevo_header_element' ) ) :
/**
 * Wrapper function to print HTML markup for all header element.
 * 
 * @param string $element
 */
function nevo_header_element( $element ) {
	// Abort if element slug is empty.
	if ( empty( $element ) ) {
		return;
	}

	// Classify element into its type.
	$type = preg_replace( '/-\d$/', '', $element );

	// Add passing variables.
	$variables = array( 'element' => $element, 'slug' => $element );

	// Get header element template.
	$html = nevo_get_template_part( 'header/' . $type, null, $variables, false );

	// Filters to modify the final HTML tag.
	$html = apply_filters( 'nevo_header_element', $html, $element );
	$html = apply_filters( 'nevo_header_element/' . $element, $html );

	echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}
endif;