<?php
/**
 * Add HTML attributes to our theme elements.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * This class adds HTML attributes to various theme elements.
 */
class Nevo_HTML_Attributes {
	/**
	 * Class instance.
	 *
	 * @access private
	 * @var $instance Class instance.
	 */
	private static $instance;

	/**
	 * Initiator
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 *  Constructor
	 */
	public function __construct() {
		add_filter( 'nevo_parse_attr', array( $this, 'parse_attributes' ), 10, 3 );
	}

	/**
	 * Parse the attributes.
	 *
	 * @since 1.0
	 * @param array  $attributes The current attributes.
	 * @param string $context The context in which attributes are applied.
	 * @param array  $settings Custom settings passed to the filter.
	 */
	public function parse_attributes( $attributes, $context, $settings ) {
		switch ( $context ) {
			case 'header':
				return $this->site_header( $attributes );

			case 'menu_toggle':
				return $this->menu_toggle( $attributes );

			case 'primary_navigation':
				return $this->primary_navigation( $attributes );

			case 'secondary_navigation':
				return $this->secondary_navigation( $attributes );
				
			case 'vertical_menu':
				return $this->vertical_menu( $attributes );
				
			case 'footer_navigation':
				return $this->footer_navigation( $attributes );

			case 'mobile_navigation':
				return $this->mobile_navigation( $attributes );
			
			case 'footer':
				return $this->footer( $attributes );

			case 'site_info':
				return $this->site_info( $attributes );

			case 'entry_header':
				return $this->entry_header( $attributes );

			case 'page_header':
				return $this->page_header( $attributes );

			case 'site_content':
				return $this->site_content( $attributes );

			case 'page':
				return $this->page( $attributes );

			case 'content':
				return $this->content( $attributes );

			case 'main':
				return $this->main( $attributes );

			case 'post_navigation':
				return $this->post_navigation( $attributes );

			case 'left_sidebar':
				return $this->left_sidebar( $attributes );

			case 'right_sidebar':
				return $this->right_sidebar( $attributes );

			case 'comment_body':
				return $this->comment_body( $attributes, $settings );

			case 'comment_meta':
				return $this->comment_meta( $attributes );

			case 'footer_entry_meta':
				return $this->footer_entry_meta( $attributes );
		}

		return $attributes;
	}

	/**
	 * Add attributes to our site header.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function site_header( $attributes ) {
		$attributes['id'] = 'masthead';
		$attributes['aria-label'] = esc_attr__( 'Site', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our menu toggle.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function menu_toggle( $attributes ) {
		$attributes['class'] .= ' menu-toggle';
		$attributes['aria-controls'] = 'primary-menu';
		$attributes['data-target'] = 'nevo-vertical-header';
		$attributes['aria-expanded'] = 'false';

		return $attributes;
	}

	/**
	 * Add attributes to our main navigation.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function primary_navigation( $attributes ) {
		$attributes['class'] .= ' nevo-navigation';
		$attributes['data-section'] = 'menu_locations';
		$attributes['id'] = 'primary-navigation';
		$attributes['aria-label'] = esc_attr__( 'Primary', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our Secondary navigation.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function secondary_navigation( $attributes ) {
		$attributes['class'] .= ' nevo-navigation';
		$attributes['data-section'] = 'menu_locations';
		$attributes['id'] = 'secondary-navigation';
		$attributes['aria-label'] = esc_attr__( 'Secondary', 'nevo' );

		return $attributes;
	}
	
	/**
	 * Add attributes to our Vertical navigation.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function vertical_menu( $attributes ) {
		$attributes['class'] .= ' nevo-navigation vertical-navigation';
		$attributes['data-section'] = 'menu_locations';
		$attributes['id'] = 'vertical-menu';
		$attributes['aria-label'] = esc_attr__( 'Vertical Menu', 'nevo' );

		return $attributes;
	}
	
	/**
	 * Add attributes to our footer navigation.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function footer_navigation( $attributes ) {
		$attributes['class'] .= ' nevo-navigation';
		$attributes['data-section'] = 'menu_locations';
		$attributes['id'] = 'footer-navigation';
		$attributes['aria-label'] = esc_attr__( 'Footer', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our mobile navigation.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function mobile_navigation( $attributes ) {
		$attributes['class'] .= ' nevo-navigation';
		$attributes['data-section'] = 'menu_locations';
		$attributes['id'] = 'mobile-navigation';
		$attributes['aria-label'] = esc_attr__( 'Mobile', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our footer element.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function footer( $attributes ) {
		$attributes['class'] .= ' site-footer';
		$attributes['id'] = 'site-footer';

		return $attributes;
	}

	/**
	 * Add attributes to our footer element.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function site_info( $attributes ) {
		$attributes['class'] .= ' site-info';
		$attributes['aria-label'] = esc_attr__( 'Site', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our entry headers.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function entry_header( $attributes ) {
		$attributes['class'] .= ' entry-header';
		$attributes['aria-label'] = esc_attr__( 'Content', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our page headers.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function page_header( $attributes ) {
		$attributes['class'] .= ' page-header';
		$attributes['id'] = 'page-header';
		$attributes['aria-label'] = esc_attr__( 'Page', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our entry headers.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function post_navigation( $attributes ) {
		if ( is_single() ) {
			$attributes['class'] .= ' post-navigation';
			$attributes['aria-label'] = esc_attr__( 'Single Post', 'nevo' );
		} else {
			$attributes['class'] .= ' paging-navigation cv-mb-0';
			$attributes['aria-label'] = esc_attr__( 'Archive Page', 'nevo' );
		}

		return $attributes;
	}

	/**
	 * Add attributes to our page container.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function page( $attributes ) {
		$attributes['id'] = 'page';
		$attributes['class'] .= ' site';

		return $attributes;
	}

	/**
	 * Add attributes to our site content container.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function site_content( $attributes ) {
		$attributes['id'] = 'content';
		$attributes['class'] .= ' site-content nevo-container position-relative';

		return $attributes;
	}

	/**
	 * Add attributes to our primary content container.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function content( $attributes ) {
		$attributes['id'] = 'primary';

		return $attributes;
	}

	/**
	 * Add attributes to our primary content container.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function main( $attributes ) {
		$attributes['id'] = 'main';

		return $attributes;
	}

	/**
	 * Add attributes to our left sidebar.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function left_sidebar( $attributes ) {
		
		$attributes['id'] = 'left-sidebar';

		return $attributes;
	}

	/**
	 * Add attributes to our right sidebar.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function right_sidebar( $attributes ) {
		
		$attributes['id'] = 'right-sidebar';

		return $attributes;
	}

	/**
	 * Add attributes to our footer widget inner container.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 * @param array $settings Settings passed through the function.
	 */
	public function comment_body( $attributes, $settings ) {
		$attributes['class'] .= ' comment-body';
		$attributes['class'] .= ' d-flex clearfix';
		$attributes['id'] = 'div-comment-' . $settings['comment-id'];

		return $attributes;
	}

	/**
	 * Add attributes to our comment meta.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function comment_meta( $attributes ) {
		$attributes['class'] .= ' comment-meta';
		$attributes['class'] .= ' d-flex align-items-center clearfix';
		$attributes['aria-label'] = esc_attr__( 'Comment meta', 'nevo' );

		return $attributes;
	}

	/**
	 * Add attributes to our footer entry meta.
	 *
	 * @since 1.0
	 * @param array $attributes The existing attributes.
	 */
	public function footer_entry_meta( $attributes ) {
		$attributes['class'] .= ' entry-meta';
		$attributes['aria-label'] = esc_attr__( 'Entry meta', 'nevo' );

		return $attributes;
	}
}

Nevo_HTML_Attributes::get_instance();
