<?php
/**
 * General functions.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'nevo_page_header_section' ) ) {
	/**
	 * Build the page header.
	 *
	 * @since 1.0.0
	 */
	function nevo_page_header_section() {
		$disable = false;
		$individual = wp_parse_args( get_post_meta( get_the_ID(), '_nevo_page_settings', true ), array() );
		$layout = isset($individual['post_single_layout']) ? $individual['post_single_layout'] : Nevo()->get_setting( 'post_single_layout' );
		$disable_b = nevo_get_page_setting( 'disable_breadcrumb' );
		$breadcrumb = wp_parse_args( get_post_meta( get_the_ID(), '_nevo_page_settings', true ) );// || isset( $breadcrumb['disable_breadcrumb'] )
		
		if ( 1 !== intval( Nevo()->get_setting( 'page_header_display' ) ) ) {
			$disable = true;
		}
		if ( is_single() && $disable_b && ('single-post-1' === $layout || 'single-post-7' === $layout) ) {
			$disable = true;
		}
		if ( $disable ) {
			$display = nevo_get_page_setting( 'page_header_display' );
			
			if ( $display === 'disable' || $display === 1 ) {
				$disable = true;
			}
		}
		if ( apply_filters( 'nevo_disable_page_header', $disable ) || is_home() || is_404() ) {
			return;
		}
		$style = '';
		$bg_image = nevo_get_page_header_background_image();
		
		$elements  = nevo_get_page_header_elements();
		
		if ( $bg_image ) {
			$style = ' style="background-image: url(' . $bg_image . ')" ';
		}
		$layout = nevo_get_page_setting( 'page_header_layout' );
		?>
		<div <?php nevo_do_attr( 'page_header' ); ?><?php echo $style; ?>>
			<div class="nevo-container inside-page-header <?php echo esc_attr( $layout ); ?>"> 
				<?php
				// Print all elements.
				foreach ( $elements as $element ) {
					$element = wp_parse_args(
						$element,
						array(
							'title'     => '',
							'elements'  => '',
							'_visibility' 	=> '',
						)
					);
					if ( 'hidden' !== $element['_visibility'] ) {
						nevo_do_page_header_element( $element['elements'] );
					}
				}
				?>
			</div>
		</div><?php
	}
}

if ( ! function_exists( 'nevo_get_page_header_elements' ) ) {
	/**
	 * Get the page header elements.
	 *
	 * @since 1.0.0
	 */
	function nevo_get_page_header_elements() {
		
		$elements  = apply_filters( 'nevo_get_page_header_elements', null );
		if ( ! $elements ) {
			if ( is_page() ) { // Page.
				$elements = Nevo()->get_setting( 'page_single_page_header_elements' );
			} elseif ( is_search() ) { // Search.
				$elements = Nevo()->get_setting( 'search_page_header_elements' );
			} elseif ( is_archive() || is_category() || is_tag() ) { // Archive.
				$elements = Nevo()->get_setting( 'post_archive_page_header_elements' );
			} elseif ( is_singular( 'post' ) ) { // single page.
				$elements = post_single_page_header_elements( $elements );
			} elseif ( is_singular() ) {
				$elements = Nevo()->get_setting( get_post_type() . '_single_page_header_elements' );
			}
		}		
		return apply_filters( 'nevo_page_header_elements', $elements );
	}
}

if ( ! function_exists( 'nevo_get_page_header_background_image' ) ) {
	/**
	 * Get the page header background.
	 *
	 * @since 1.0.0
	 */
	function nevo_get_page_header_background_image() {
		$bg_image = apply_filters( 'nevo_get_page_header_background_image', null );
		$image = '';
		if ( ! $bg_image ) {
			if ( is_page() ) { // Page.
				$image = Nevo()->get_setting( 'page_single_page_header_background_image' );
			} elseif ( is_single() ) {
				$layout = Nevo()->get_setting( 'post_single_layout' );
				if ( 'single-post-4' === $layout || 'single-post-5' === $layout || 'single-post-6' === $layout ) {
					$image = 'thumbnail';
				}
			} elseif ( is_singular() ) {
				$image = Nevo()->get_setting( get_post_type() . '_single_page_header_background_image' );
			}
			if ( $image === 'thumbnail' ) {
				$bg_image = get_the_post_thumbnail_url( get_the_ID(), 'full' );
			}
		}
		
		return apply_filters( 'nevo_page_header_background_image', $bg_image );
	}
}

if ( ! function_exists( 'nevo_do_page_header_element' ) ) {
	/**
	 * Render page header element.
	 */
	function nevo_do_page_header_element( $element ) {
		if ( empty( $element ) ) {
			return;
		}
		if ( 'title' === $element ) {
			if ( is_singular() ) {
				ob_start();
				$html = nevo_display_entry_header();
				$html = ob_get_clean();
			} elseif ( is_search() ) {
				$html = nevo_do_search_results_title();
			} else {
				ob_start();
				nevo_archive_title();
				$html = ob_get_clean();
			}
		}
		if ( 'featured-image' === $element ) {
			ob_start();
			nevo_featured_inside_single();
			$html = ob_get_clean();
		}
		if ( 'search-form' === $element ) {
			$html = get_search_form();
		}
		if ( 'breadcrumb' === $element ) {
			$html = nevo_breadcrumb( false );
		}
		if ( 'description' === $element ) {
			ob_start();
			nevo_do_archive_description();
			$html = ob_get_clean();
		}
		// Filters to modify the final HTML tag.
		$html = apply_filters( 'nevo/page_header_element', $html, $element );
		$html = apply_filters( 'nevo/page_header_element/' . $element, $html );
				
		echo $html;
	}
}

add_action( 'wp', 'nevo_template_hooks', 5 );
/**
 * Attach template functions into the proper template hooks.
 *
 * @since 1.0
 */
function nevo_template_hooks() {
	
	$header_post_types = apply_filters(
		'nevo_entry_meta_post_types',
		array(
			'post',
		)
	);

	if ( in_array( get_post_type(), $header_post_types ) ) {
		add_action( 'nevo_after_entry_title', 'nevo_post_meta', 10 );
	}

	$footer_post_types = apply_filters(
		'nevo_footer_meta_post_types',
		array(
			'post',
		)
	);

	if ( in_array( get_post_type(), $footer_post_types ) && is_singular() ) {
		add_action( 'nevo_after_entry_content', 'nevo_footer_meta', 10 );
	}
	
	$categories_post_types = apply_filters(
		'nevo_categories_entry_meta_post_types',
		array(
			'post',
		)
	);
	
	if ( in_array( get_post_type(), $categories_post_types ) && is_singular() ) {
		add_action( 'nevo_before_entry_title', 'nevo_categories_meta', 10 );
	}
	if ( in_array( get_post_type(), $categories_post_types ) ) {
		add_action( 'nevo_flexible_category_meta', 'nevo_categories_meta', 10 );
	}
	
	$individual = wp_parse_args( get_post_meta( get_the_ID(), '_nevo_page_settings', true ), array() );
	$layout = isset($individual['post_single_layout']) ? $individual['post_single_layout'] : Nevo()->get_setting( 'post_single_layout' );
	
	$display_set  = apply_filters( 'nevo_set_page_header_display', false );
	$display = nevo_get_page_setting( 'page_header_display' );
	
	if ( is_single() && $display === 'before' && ( 'single-post-1' === $layout || 'single-post-7' === $layout ) ) {
		add_action( 'nevo_before_while', 'nevo_page_header_section', 10 );
	} elseif ( ! is_single() && ( $display === 'before' && ! $display_set ) || $display_set ) {
		add_action( 'nevo_before_while', 'nevo_page_header_section', 10 );
	}
	if ( $display === 'after' && ! $display_set ) {
		add_action( 'nevo_after_header', 'nevo_page_header_section', 10 );
	}
}

if ( ! function_exists( 'nevo_breadcrumb' ) ) {
	/**
	 * Print / return HTML markup for breadcrumb.
	 *
	 * @param boolean $echo
	 * @return string
	 */
	function nevo_breadcrumb( $echo = true ) {
		$html = '';

		// Allow breadcrumb markup modification before using the normal procedures.
		$html = apply_filters( 'nevo_breadcrumb_html', $html );
		
		// If markup is still empty, proceed to the normal procedures.
		if ( empty( $html ) ) {
			// Get breadcrumb markup based on the breadcrumb module mode.
			switch ( Nevo()->get_setting( 'global_breadcrumb_type' ) ) {
				case 'breadcrumb-trail':
					if ( function_exists( 'breadcrumb_trail' ) ) {
						$html = breadcrumb_trail( array(
							'show_browse' => false,
							'echo' => false,
						) );
					}
					break;
		
				case 'breadcrumb-navxt':
					if ( function_exists( 'bcn_display' ) ) {
						$html = bcn_display( true );
					}
					break;
		
				case 'yoast-seo':
					if ( function_exists( 'yoast_breadcrumb' ) ) {
						$html = yoast_breadcrumb( '', '', false );
					}
					break;
		
				case 'rank-math':
					if ( function_exists( 'rank_math_get_breadcrumbs' ) ) {
						$html = rank_math_get_breadcrumbs();
					}
					break;
		
				case 'seopress':
					if ( function_exists( 'seopress_display_breadcrumbs' ) ) {
						$html = seopress_display_breadcrumbs( false );
					}
					break;
		
				default:
					$html = nevo_breadcrumb_native( false );
					break;
			}
		}

		// Wrap with "nevo-breadcrumb" div.
		$html = '<div class="nevo-breadcrumb">' . $html . '</div>';

		// Render or return the HTML markup.
		$disable = nevo_get_page_setting( 'disable_breadcrumb' );
		
		if ( $disable ) {
			return;
		}
		if ( $echo ) {
			echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'nevo_breadcrumb_native' ) ) {
	/**
	 * Render theme's native breadcrumb.
	 *
	 * @param boolean $echo
	 * @return string
	 */
	function nevo_breadcrumb_native( $echo = true ) {
		$items = array();

		/**
		 * Build breadcrumb trails.
		 */

		// Home
		if ( intval(  Nevo()->get_setting( 'global_breadcrumb_home' ) ) ) {
			$items['home'] = array(
				'label' => apply_filters( 'nevo_breadcrumb_home_label', esc_html__( 'Home', 'nevo' ) ),
				'url'   => apply_filters( 'nevo_breadcrumb_home_url', home_url( '/' ) ),
			);
		}

		// Search results page
		if ( is_search() ) {
			$items['search'] = array(
				/* translators: %s: search keyword. */
				'label' => sprintf( esc_html__( 'Search: %s', 'nevo' ), get_search_query() ),
			);
		}

		// 404 page
		elseif ( is_404() ) {
			$items['404'] = array(
				'label' => esc_html__( 'Page Not Found', 'nevo' ),
			);
		}

		// Other kind of archives: taxonomy archive.
		elseif ( is_archive() || is_home() ) {
			$post_type = get_post_type();
			$post_type_obj = get_post_type_object( $post_type );
			
			// Add post type archive page if it's not same with home page.
			if ( ! is_front_page() && trailingslashit( get_post_type_archive_link( $post_type ) ) !== home_url( '/' ) ) {
				$post_type_obj = get_post_type_object( $post_type );
				if ( is_object( $post_type_obj ) ) {
					$items['post_type_archive'] = array(
						'label' => 'post' === $post_type ? get_the_title( get_option( 'page_for_posts' ) ) : $post_type_obj->labels->name,
						'url'   => is_post_type_archive() || is_home() ? '' : get_post_type_archive_link( $post_type ),
					);
				}
			}

			// Date archive
			if ( is_date() ) {
				// Add published year info for year archive, month archive, and day archive.
				if ( is_year() || is_month() || is_day() ) {
					$items['year'] = array(
						'label' => get_the_time( 'Y' ),
						'url'   => is_year() ? '' : get_year_link( get_the_time( 'Y' ) ),
					);
				}
				// Add published month info for month archive, and day archive.
				if ( is_month() || is_day() ) {
					$items['month'] = array(
						'label' => get_the_time( 'F' ),
						'url'   => is_month() ? '' : get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ),
					);
				}
				// Add published day info for day archive.
				if ( is_day() ) {
					$items['day'] = array(
						'label' => get_the_time( 'd' ),
					);
				}
			}
			// Author archive
			elseif ( is_author() ) {
				$author = get_userdata( get_query_var( 'author' ) );

				$items['author_archive'] = array(
					/* translators: %s: author display name. */
					'label' => $author->display_name,
				);
			}
			// Taxonomy archive
			elseif ( is_category() || is_tag() || is_tax() ) {
				$term = get_queried_object();
				$tax = get_taxonomy( $term->taxonomy );
				$parents = get_ancestors( $term->term_id, $term->taxonomy );

				$i = count( $parents );

				while ( $i > 0 ) {
					$parent_term = get_term( $parents[ $i - 1 ], $term->taxonomy );

					$items['term_parent__' . $i ] = array(
						'label' => $parent_term->name,
						'url'   => get_term_link( $parent_term, $parent_term->taxonomy ),
					);

					$i--;
				}
				$items['term'] = array(
					'label' => $term->name,
				);
			}
		}

		// All singular types
		elseif ( is_singular() && ! is_front_page() ) {
			global $post;
			
			// All singular types except attachments
			if ( ! is_attachment() ) {
				$post_type = get_post_type();

				// Post type archive link for Post and other CPT.
				if ( is_single() ) {
					// Add post type archive page if it's not same with home page.
					if ( trailingslashit( get_post_type_archive_link( $post_type ) ) !== home_url( '/' ) ) {
						$post_type_obj = get_post_type_object( $post_type );

						$items['post_type_archive'] = array(
							'label' => 'post' === $post_type ? get_the_title( get_option( 'page_for_posts' ) ) : $post_type_obj->labels->name,
							'url'   => get_post_type_archive_link( $post_type ),
						);
					}
				}
				
				// Category trails for Post.
				if ( 'post' === $post_type ) {
					$cats = get_the_category();
					$cat_id = $cats[0]->term_id;
					$parents = get_ancestors( $cat_id, 'category' );

					$i = count( $parents );

					while ( $i > 0 ) {
						$items['term_parent__' . $i ] = array(
							'label' => get_cat_name( $parents[ $i - 1 ] ),
							'url'   => get_category_link( $parents[ $i - 1 ] ),
						);

						$i--;
					}

					$items['term'] = array(
						'label' => get_cat_name( $cat_id ),
						'url'   => get_category_link( $cat_id ),
					);
				}

				// Ancestors Trails for Page and other CPT.
				$ancestors = get_post_ancestors( get_post() );
				$i = count( $ancestors );

				while ( $i > 0 ) {
					$items['post_parent__' . $i ] = array(
						'label' => get_the_title( $ancestors[ $i - 1 ] ),
						'url'   => get_permalink( $ancestors[ $i - 1 ] ),
					);

					$i--;
				}
			}

			// Current singular page.
			$items['post'] = array(
				'label' => get_the_title(),
			);
		}

		// Paginated
		$paged = get_query_var( 'page' ) || get_query_var( 'paged' );
		$keys = array_keys( $items );
		$last_key = end( $keys );

		if ( $paged ) {
			/* translators: %s: page number. */
			$items[ $last_key ]['label'] .= sprintf( esc_html__( ' (Page %d)', 'nevo' ), get_query_var( 'paged' ), $paged );
		}

		// Remove last item in the trail if "current page" is set to hidden.
		if ( ! intval( Nevo()->get_setting( 'global_breadcrumb_current_page' ) ) ) {
			unset( $items[ $last_key ] );
		}

		// Allow developers to modify the breadcrumb trail.
		$items = apply_filters( 'nevo_breadcrumb_trail_items', $items );

		// Abort if there is no trail.
		if ( empty( $items ) ) {
			return;
		}

		// Abort if breadcrumb trail only contain 1 item and the "Hide if home or current page is the only item" mode is enabled.
		if ( 1 == count( $items ) && intval( Nevo()->get_setting( 'global_breadcrumb_home_or_current' ) ) ) {
			// If home or current page (doesn't contain URL).
			if ( 'home' === array_key_first( $items ) || ! isset( $items[0]['url'] ) ) {
				return;
			}
		}
		/**
		 * Render breadcrumb markup.
		 */
		$item_count = count( $items );
		
		// Opening tag.
		$html = '<ul class="nevo-breadcrumb-native d-inline-block cv-mb-0 cv-ps-0" itemscope itemtype="https://schema.org/BreadcrumbList">';
		
		// Build breadcrumb markup.
		$separator = Nevo()->get_setting( 'global_breadcrumb_separator' );
		$separator = ' <span class="separator">'. $separator .'</span> ';
		
		$item_position = 0;
		foreach ( $items as $index => $item ) {
			++$item_position;
			
			$item_class = 'breadcrumb-item d-inline-block';
			
			if ( 1 === $item_position && 1 < $item_count ){
				$item_class .= ' item-first';
			}elseif ( $item_count === $item_position ){
				$item_class .= ' item-last';
			}
			// Separator
			if ( $item_count === $item_position ) {
				$sep = '';
			} else {
				$sep = $separator;
			}
			$html .= '<li class="' . esc_attr( $item_class ) . '" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">';

				if ( isset( $item['url'] ) && ! empty( $item['url'] ) ) {
					$html .= '<a itemprop="item" href="' . esc_url( $item['url'] ) . '"><span itemprop="name">' . esc_html( $item['label'] ) . '</span></a>';
				} else {
					$html .= '<span itemprop="name">' . esc_html( $item['label'] ) . '</span>';
				}
				$html .= '<meta itemprop="position" content="' . esc_attr( $item_position ) . '" />';
				$html .= $sep;

			$html .= '</li>';
		}

		// Closing tag.
		$html .= '</ul>';

		// Render or return the HTML markup.
		if ( $echo ) {
			echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			return $html;
		}
	}
}

add_action( 'nevo_content_entry', 'nevo_do_post_loop', 5 );
/**
 * Build the Post loop.
 *
 * @since 1.0.0
 */
function nevo_do_post_loop() {
	if ( nevo_blog_columns() ) {
		nevo_get_template_part('blog/'. nevo_get_post_setting('layout') );
		//nevo_get_template_part('blog/content-29' );
	}
}

if ( ! function_exists( 'nevo_do_content_loop' ) ) {
	add_action( 'nevo_content_entry', 'nevo_do_content_loop', 10 );
	/**
	 * do content loop on page and on post page.
	 *
	 * @since 1.0
	 */
	function nevo_do_content_loop() {
		if ( nevo_blog_columns() ) {
			return;
		}
		?>
		<article id="post-<?php the_ID(); ?>" <?php post_class(); ?> <?php nevo_do_microdata( 'article' ); ?>>
			<div class="inside-article">
				<?php
				/**
				 * nevo_before_content hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_before_content' );
				
				/**
				 * nevo_before_entry_header hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_before_entry_header' );
				
				// Move entry header.
				$post_type = Nevo()->get_setting( get_post_type() . '_single_page_header_elements' );
				if ( is_array( $post_type ) ) {
				  $post_type = nevo_create_visibility_elements( $post_type );
				}
				$page = nevo_create_visibility_elements( Nevo()->get_setting( 'page_single_page_header_elements' ) );
				
				$layout = Nevo()->get_setting( 'post_single_layout' );
				if ( ( is_page() && !$page ) || ( is_single() && ('single-post-1' === $layout
					|| 'single-post-6' === $layout || 'single-post-7' === $layout) ) ) {
					nevo_display_entry_header();
				} elseif ( !is_page() && !is_single() && is_singular() && !$post_type ) {
					nevo_display_entry_header();
				} elseif ( !is_singular() ) {
					nevo_display_entry_header();
				}
				
				/**
				 * nevo_after_entry_header hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_after_entry_header' );

				/**
				 * nevo_before_entry_content hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_before_entry_content' );
				
				nevo_get_entry_content('', $type = 'content');

				/**
				 * nevo_after_entry_content hook.
				 *
				 * @since 1.0
				 *
				 * @hooked nevo_footer_meta - 10
				 */
				do_action( 'nevo_after_entry_content' );

				/**
				 * nevo_after_content hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_after_content' );
				?>
			</div>
		</article>
		<?php
	}
}

if ( ! function_exists( 'nevo_show_title' ) ) {
	/**
	 * Check to see if we should show our page/post title or not.
	 *
	 * @since 1.0
	 *
	 * @return bool Whether to show the content title.
	 */
	function nevo_show_title() {
		return apply_filters( 'nevo_show_title', true );
	}
}

/**
 * Check whether we should display the entry header or not.
 *
 * @since 1.0
 */
function nevo_display_entry_header() {
	$show_entry_header      = true;
	$show_title             = nevo_show_title();
	$has_before_entry_title = has_action( 'nevo_before_entry_title' );
	$has_after_entry_title  = has_action( 'nevo_after_entry_title' );

	if ( ! $show_title && ! $has_before_entry_title && ! $has_after_entry_title ) {
		$show_entry_header = false;
	}
	
	if ( apply_filters( 'nevo_display_entry_header', $show_entry_header ) ) :
		?>
		<header <?php nevo_do_attr( 'entry_header' ); ?>>
			<?php
			/**
			 * nevo_before_entry_title hook.
			 *
			 * @since 1.0
			 *
			 * @hooked nevo_categories_meta - 10
			 */
			do_action( 'nevo_before_entry_title' );

			if ( nevo_show_title() ) {
				$params = nevo_get_the_title_parameters();

				the_title( $params['before'], $params['after'] );
			}

			/**
			 * nevo_after_entry_title hook.
			 *
			 * @since 1.0
			 *
			 * @hooked nevo_post_meta - 10
			 */
			do_action( 'nevo_after_entry_title' );
			?>
		</header>
		<?php
	endif;
}

/**
 * Build our the_title() parameters.
 *
 * @since 1.0
 */
function nevo_get_the_title_parameters($extra_classes = '') {
	$params = array(
		'before' => sprintf(
			'<h1 class="entry-title%1$s"%2$s>',
			esc_attr($extra_classes),
			'microdata' === nevo_get_schema_type() ? ' itemprop="headline"' : ''
		),
		'after' => '</h1>',
	);

	if ( ! is_singular() ) {
		$params = array(
			'before' => sprintf(
				'<h2 class="entry-title%1$s"%3$s><a href="%2$s" title="' . the_title_attribute( array( 'echo' => false ) ) . '" rel="bookmark">',
				esc_attr($extra_classes),
				esc_url( get_permalink() ),
				'microdata' === nevo_get_schema_type() ? ' itemprop="headline"' : ''
			),
			'after' => '</a></h2>',
		);
	}

	if ( 'link' === get_post_format() ) {
		$params = array(
			'before' => sprintf(
				'<h2 class="entry-title%1$s"%3$s><a href="%2$s" title="' . the_title_attribute( array( 'echo' => false ) ) . '" rel="bookmark">',
				esc_attr($extra_classes),
				esc_url( nevo_get_link_url() ),
				'microdata' === nevo_get_schema_type() ? ' itemprop="headline"' : ''
			),
			'after' => '</a></h2>',
		);
	}

	return apply_filters( 'nevo_get_the_title_parameters', $params );
}

if ( ! function_exists( 'nevo_get_entry_content' ) ) {
	/**
	 * Figure out if we should show the blog excerpts or full posts
	 *
	 * @since 1.0
	 */
	function nevo_get_entry_content($length = '', $type = '', $excerpt_length = '', $extra_classes = '') {
		global $post;

		$itemprop = '';

		if ('microdata' === nevo_get_schema_type()) {
			$itemprop = ' itemprop="text"';
		}
		$excerpt_more_link = apply_filters('nevo_display_read_more', false);
		$more = apply_filters('nevo_excerpt_more', 'Read More');
		$title_attribute = the_title_attribute(['echo' => false]);
		$more_link = $excerpt_more_link ? ' <p class="excerpt-more d-block"><a title="' . $title_attribute . '" class="read-more-button button" href="' . get_permalink() . '" rel="bookmark">' . esc_attr($more) . '<span class="screen-reader-text">' . $title_attribute . '</span></a></p>' : '';

		if ('custom' == $type) {
			$length = $excerpt_length ? $excerpt_length : $length;
		}
		$length = apply_filters('nevo_excerpt_length', $length);

		$excerpt = wp_trim_words(strip_shortcodes($post->post_content), $length);

		if ('content' == $type) {
			$entry_content = '<div class="entry-content" ' . $itemprop . '>';
			$entry_content .= apply_filters('the_content', get_the_content());
			$entry_content .= wp_link_pages(
				array(
					'before' => '<div class="page-links">' . __('Pages:', 'nevo'),
					'after' => '</div>',
				)
			);
			$entry_content .= '</div>';
			echo $entry_content;

		} elseif ('default' == $type || $excerpt_length > 0) {
			echo '<div class="entry-summary' . esc_attr($extra_classes) . '" ' . $itemprop . '><p>' . esc_attr($excerpt) . '</p>' . $more_link . '</div>';
		}
	}

}
		
if ( ! function_exists( 'nevo_get_link_url' ) ) {
	/**
	 * Return the post URL.
	 *
	 * Falls back to the post permalink if no URL is found in the post.
	 *
	 * @since 1.0
	 *
	 * @see get_url_in_content()
	 * @return string The Link format URL.
	 */
	function nevo_get_link_url() {
		$has_url = get_url_in_content( get_the_content() );

		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core filter name.
		return $has_url ? $has_url : apply_filters( 'the_permalink', get_permalink() );
	}
}

/**
 * Gets the SVG code for a given icon.
 *
 * @param string $group The icon group.
 * @param string $icon The icon.
 * @param int    $size The icon size in pixels.
 *
 * @return string
 */
function nevo_get_svg_icon( $group, $icon, $classes, $size = 16 ) {
	return Nevo_SVG_Icons::get_svg( $group, $icon, $classes, $size );
}

/**
 * Get our media queries.
 *
 * @since 1.0
 *
 * @param string $name Name of the media query.
 * @return string The full media query.
 */
function nevo_get_media_query( $name ) {
	$desktop = apply_filters( 'nevo_desktop_media_query', '@media (min-width:992px){' );
	$tablet = apply_filters( 'nevo_tablet_media_query', '@media (min-width: 576px) and (max-width: 991.98px){' );
	$phone = apply_filters( 'nevo_phone_media_query', '@media (max-width:575.98px){' );

	$queries = apply_filters(
		'nevo_media_queries',
		array(
			'desktop' => $desktop,
			'tablet' => $tablet,
			'mobile' => $phone,
		)
	);

	return $queries[ $name ];
}

if ( ! function_exists( 'nevo_smart_content_width' ) ) {
	add_action( 'wp', 'nevo_smart_content_width' );
	/**
	 * Set the $content_width depending on layout of current page
	 * Hook into "wp" so we have the correct layout setting
	 * Hooking into "after_setup_theme" doesn't get the correct layout setting
	 */
	function nevo_smart_content_width() {
		global $content_width;

		$container_width = Nevo()->get_setting( 'container_width' );
		$container_width = wp_parse_args(
			$container_width,
			array(
				'unit' => '',
				'value' => '1200',
			)
		);
		$sidebar_width = Nevo()->get_setting( 'sidebar_width' );
		$sidebar_width = wp_parse_args(
			$sidebar_width,
			array(
				'unit' => '',
				'value' => '30',
			)
		);
		$container_width = $container_width['value'];
		$right_sidebar_width = $sidebar_width['value'];
		$left_sidebar_width = $sidebar_width['value'];
		$layout = nevo_get_page_setting('sidebar_layout');

		if ( 'left-sidebar' === $layout ) {
			$content_width = $container_width * ( ( 100 - $left_sidebar_width ) / 100 );
		} elseif ( 'right-sidebar' === $layout ) {
			$content_width = $container_width * ( ( 100 - $right_sidebar_width ) / 100 );
		} elseif ( 'no-sidebar' === $layout ) {
			$content_width = $container_width;
		} else {
			$content_width = $container_width * ( ( 100 - ( $left_sidebar_width + $right_sidebar_width ) ) / 100 );
		}
	}
}

if ( ! function_exists( 'nevo_disable_title' ) ) {
	add_filter( 'nevo_show_title', 'nevo_disable_title' );
	/**
	 * Remove our title if set.
	 *
	 * @since 1.0
	 *
	 * @param bool $title Whether the title is displayed or not.
	 * @return bool Whether to display the content title.
	 */
	function nevo_disable_title( $title ) {
		if ( is_singular() ) {
			$disable_title = wp_parse_args( get_post_meta( get_the_ID(), '_nevo_page_settings', true ) );

			if ( isset( $disable_title['disable_content_title'] ) ) {
				$title = false;
			}
		}

		return $title;
	}
}

if ( ! function_exists( 'nevo_remove_caption_padding' ) ) {
	add_filter( 'img_caption_shortcode_width', 'nevo_remove_caption_padding' );
	/**
	 * Remove WordPress's default padding on images with captions
	 *
	 * @param int $width Default WP .wp-caption width (image width + 10px).
	 * @return int Updated width to remove 10px padding.
	 */
	function nevo_remove_caption_padding( $width ) {
		return $width - 10;
	}
}

if ( ! function_exists( 'nevo_enhanced_image_navigation' ) ) {
	add_filter( 'attachment_link', 'nevo_enhanced_image_navigation', 10, 2 );
	/**
	 * Filter in a link to a content ID attribute for the next/previous image links on image attachment pages.
	 *
	 * @param string $url The input URL.
	 * @param int    $id The ID of the post.
	 */
	function nevo_enhanced_image_navigation( $url, $id ) {
		if ( ! is_attachment() && ! wp_attachment_is_image( $id ) ) {
			return $url;
		}

		$image = get_post( $id );
		// phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison -- Intentially loose.
		if ( ! empty( $image->post_parent ) && $image->post_parent != $id ) {
			$url .= '#main';
		}

		return $url;
	}
}

if ( ! function_exists( 'nevo_categorized_blog' ) ) {
	/**
	 * Determine whether blog/site has more than one category.
	 *
	 * @since 1.0
	 *
	 * @return bool True of there is more than one category, false otherwise.
	 */
	function nevo_categorized_blog() {
		if ( false === ( $all_the_cool_cats = get_transient( 'nevo_categories' ) ) ) { // phpcs:ignore
			// Create an array of all the categories that are attached to posts.
			$all_the_cool_cats = get_categories(
				array(
					'fields'     => 'ids',
					'hide_empty' => 1,

					// We only need to know if there is more than one category.
					'number'     => 2,
				)
			);

			// Count the number of categories that are attached to the posts.
			$all_the_cool_cats = count( $all_the_cool_cats );

			set_transient( 'nevo_categories', $all_the_cool_cats );
		}

		if ( $all_the_cool_cats > 1 ) {
			// This blog has more than 1 category so twentyfifteen_categorized_blog should return true.
			return true;
		} else {
			// This blog has only 1 category so twentyfifteen_categorized_blog should return false.
			return false;
		}
	}
}

if ( ! function_exists( 'nevo_category_transient_flusher' ) ) {
	add_action( 'edit_category', 'nevo_category_transient_flusher' );
	add_action( 'save_post', 'nevo_category_transient_flusher' );
	/**
	 * Flush out the transients used in {@see nevo_categorized_blog()}.
	 *
	 * @since 1.0.0
	 */
	function nevo_category_transient_flusher() {
		// Like, beat it. Dig?
		delete_transient( 'nevo_categories' );
	}
}

add_filter( 'wp_headers', 'nevo_set_wp_headers' );
/**
 * Set any necessary headers.
 *
 * @param array $headers The existing headers.
 *
 * @since 1.0
 */
function nevo_set_wp_headers( $headers ) {
	$headers['X-UA-Compatible'] = 'IE=edge';

	return $headers;
}