<?php
/**
 * Blog functions.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Initiate columns.
 *
 * @since 1.0
 */
function nevo_blog_columns() {
	$columns = false;
	if ( is_home() || is_category() || is_tag() || is_date() || is_author() ) {
		$columns = true;
	}
	return apply_filters( 'nevo_blog_columns', $columns );
}

if ( ! function_exists( 'nevo_post_image' ) ) {
	/**
	 * Prints the Post Image to post excerpts
	 */
	function nevo_post_image($size = 'medium', $div_classes = '', $img_classes = '', $style = '', $category = true) {
		// Don't run the function unless we have a post thumbnail.
		if ( ! has_post_thumbnail() ) {
			return;
		}
		if ( ! is_404() ) {
			$div_classes = 'post-image size-full position-relative ' . $div_classes;
			$attrs = apply_filters('nevo_post_image_default_attribute', array(
				//'alt' => the_title_attribute('echo=0'),
				'itemprop' => 'image',
				'class' => $img_classes,
				'style' => $style
			));
			?>
			<div class="<?php echo esc_attr($div_classes); ?>">
				<?php if ($category) { do_action('nevo_flexible_category_meta'); } ?>
				<a href="<?php the_permalink(); ?>" title="<?php echo the_title_attribute('echo=0') ?>" rel="bookmark">
					<?php echo get_the_post_thumbnail(get_the_ID(),apply_filters('nevo_post_image_size', $size), $attrs); ?>
				</a>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'nevo_featured_image' ) ) {
	/**
	 * Build the page header.
	 *
	 * @since 1.0.0
	 *
	 * @param string $class The featured image container class.
	 */
	function nevo_featured_image( $class ) {
		// Don't run the function unless we have a post thumbnail.
		if ( ! has_post_thumbnail() ) {
			return;
		}

		// Don't run the function unless we're on a page it applies to.
		if ( is_singular() ) {
			$attrs = apply_filters( 'nevo_featured_image_default_attribute', array(
				'alt' => the_title_attribute( 'echo=0' ),
				'itemprop' => 'image',
				'class' => 'size-full cv-col-12',
				'style' => 'aspect-ratio:16/9;'
			) );
			?>
			<div class="featured-image cv-col-12 <?php echo esc_attr( $class ); ?>">
				<?php
					the_post_thumbnail(
						apply_filters( 'nevo_featured_image_default_size', 'full' ),
						$attrs
					);
				?>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'nevo_featured_inside_single' ) ) {
	add_action( 'nevo_after_entry_header', 'nevo_featured_inside_single', 10 );
	/**
	 * Only add to single post.
	 *
	 * @since 1.0.0
	 */
	function nevo_featured_inside_single() {
		if ( is_single() ) {
			nevo_featured_image( 'post-image' );
		}
	}
}

if ( ! function_exists( 'nevo_display_post_single_elements' ) ) {
	
	add_action( 'nevo_after_entry', 'nevo_display_post_single_elements', 10 );
	/**
	 * Build the single post elements.
	 *
	 * @since 1.0.0
	 */
	function nevo_display_post_single_elements() {
		if ( ! is_single() || ! Nevo()->get_setting( 'post_single_elements', 'elements' ) ) {
			return;
		}
		echo '<div class="single-after-entry">';
		
		do_action( 'nevo_before_content_single_elements' );
		
		foreach ( Nevo()->get_setting( 'post_single_elements' ) as $element ) {
			$element = wp_parse_args(
				$element,
				array(
					'title'       => '',
					'elements'    => '',
					'_visibility' => '',
				)
			);
			if ( 'hidden' !== $element['_visibility'] ) {
				nevo_get_template_part( 'blog/' . esc_html($element['elements']) );
			}
		}
		
		do_action( 'nevo_after_content_single_elements' );
		
		echo '</div>';
	}
}

if ( ! function_exists( 'nevo_custom_image_default_size' ) ) {
	add_filter( 'nevo_featured_image_default_size', 'nevo_custom_image_default_size' );
	/**
	 * Add size in post image size
	 *
	 * @since 1.0
	 */
	function nevo_custom_image_default_size() {
		$imagesize = Nevo()->get_setting( 'post_single_thumbnail_size' );
		
		if ( is_single() && $imagesize != 'default' ) {
			return $imagesize;
		}
		if ( is_page() ) {
			return 'full';
		}
	}
}

function post_single_page_header_elements( $elements ){
	$layout = Nevo()->get_setting( 'post_single_layout' );
	
	if ( 'single-post-1' === $layout || 'single-post-6' === $layout || 'single-post-7' === $layout ) {
		$elements = array(
			0 => array( 'elements' => 'breadcrumb' ),
		);
	}
	if ( 'single-post-2' === $layout
		|| 'single-post-4' === $layout
		|| 'single-post-5' === $layout
		|| 'single-post-8' === $layout ) {
		$elements = array(
			0 => array( 'elements' => 'breadcrumb' ),
			1 => array( 'elements' => 'title' ),
		);
	}
	if ( 'single-post-3' === $layout ) {
		$elements = array(
			0 => array( 'elements' => 'breadcrumb' ),
			1 => array( 'elements' => 'title' ),
			2 => array( 'elements' => 'featured-image' ),
		);
	}
	return apply_filters( 'post_single_page_header_elements', $elements );
}

/**
 * Collects all available image sizes which we use in the Customizer.
 *
 * @since 1.0
 *
 * @return array
 */
function nevo_blog_get_image_sizes() {
	$sizes = get_intermediate_image_sizes();

	$new_sizes = array(
		//'default'	=> 'Default',
		'full' 		=> 'full',
	);

	foreach ( $sizes as $key => $name ) {
		$new_sizes[ $name ] = $name;
	}

	return $new_sizes;
}

if ( ! function_exists( 'nevo_category_color' ) ) :

	/**
	 * Getting Category Color.
	 *
	 * @param int $wp_category_id Category id.
	 *
	 * @return string The category color.
	 */
	function nevo_category_color( $wp_category_id ) {

		$args     = array(
			'orderby'    => 'id',
			'hide_empty' => 0,
		);
		$category = get_categories( $args );

		foreach ( $category as $category_list ) {
			$color = get_theme_mod( 'nevo_category_color_' . $wp_category_id );

			return $color;
		}

	}

endif;