<?php
/**
 * Archive elements.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'nevo_archive_title' ) ) {
	/**
	 * Build the archive title
	 *
	 * @since 1.0
	 */
	function nevo_archive_title() {
		if ( ! is_archive() ) {
			return;
		}
		?>
		<div class="nevo-archive-title">
			<h1 class="page-title d-inline-block position-relative cv-mb-0">
				<?php the_archive_title(); ?>
			</h1>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_filter_the_archive_title' ) ) {
	add_filter( 'get_the_archive_title', 'nevo_filter_the_archive_title', 10, 3 );
	/**
	 * Alter the_archive_title() function to match our original archive title function
	 *
	 * @since 1.0
	 *
	 * @param string $title The archive title.
	 * @return string The altered archive title
	 */
	function nevo_filter_the_archive_title( $title, $original_title, $prefix ) {
		// Set default title for Blog page.
		if ( is_home() ) {
			// If blog page is also the front page, use tagline
			if ( is_front_page() ) {
				$title = get_bloginfo( 'description' );
			}
			// If blog page is set to static page, use the static page title.
			else {
				$title = get_the_title( get_option( 'page_for_posts' ) );
			}
		}

		// Fetch custom title that is configured from Customizer.
		if ( is_post_type_archive() || is_home() ) {
			$custom_title = 'product' === get_post_type() ? get_the_title( get_option( 'woocommerce_shop_page_id' ) ) : '{{post_type}}';
			
			if ( ! empty( $custom_title ) ) {
				$post_type_obj = get_post_type_object( get_post_type() );

				$custom_title = str_replace( '{{post_type}}', $post_type_obj->labels->name, $custom_title );
			}
		}
		elseif ( is_category() || is_tag() || is_tax() ) {
			$custom_title = '{{term}}';

			if ( ! empty( $custom_title ) ) {
				$term_obj = get_queried_object();
				$taxonomy_obj = get_taxonomy( $term_obj->taxonomy );

				$custom_title = str_replace( '{{taxonomy}}', $taxonomy_obj->labels->singular_name, $custom_title );
				$custom_title = str_replace( '{{term}}', $term_obj->name, $custom_title );
			}
		}
		elseif ( is_author() ) {
			/*
			 * Queue the first post, that way we know
			 * what author we're dealing with (if that is the case).
			 */
			the_post();

			$custom_title = sprintf(
				'<div class="author-avatar author-image alignleft cv-me-3">%1$s</div><span class="vcard text-capitalize">%2$s</span>',
				get_avatar( get_the_author_meta( 'ID' ), $size = '45', $default = '', $alt = '', $args = array( 'class' => 'author-img' ) ),
				get_the_author()
			);

			/*
			 * Since we called the_post() above, we need to
			 * rewind the loop back to the beginning that way
			 * we can run the loop properly, in full.
			 */
			rewind_posts();
		}

		// If custom title is detected, use it.
		if ( ! empty( $custom_title ) ) {
			$title = $custom_title;
		}
		
		return $title;

	}
}

/**
 * Output the archive description.
 *
 * @since 1.0
 */
function nevo_do_archive_description() {
	$term_description = get_the_archive_description();

	if ( ! empty( $term_description ) ) {
		if ( is_author() ) {
			printf( '<div class="author-info">%s</div>', $term_description ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			printf( '<div class="taxonomy-description">%s</div>', $term_description ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	/**
	 * nevo_after_archive_description hook.
	 *
	 * @since 1.0
	 */
	do_action( 'nevo_after_archive_description' );
}

/**
 * Add the search results title to the search results page.
 *
 * @since 1.0
 * @param string $template The template we're targeting.
 */
function nevo_do_search_results_title() {
	// phpcs:ignore -- No escaping needed.
	echo apply_filters(
		'nevo_search_title_output',
		sprintf(
			'<h1 class="page-title">%s</h1>',
			sprintf(
				/* translators: 1: Search query name */
				__( 'Search Results for: %s', 'nevo' ),
				'<span>' . get_search_query() . '</span>'
			)
		)
	);
}