<?php
/**
 * Comment structure.
 *
 * @package Nevo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'nevo_comment' ) ) {
	/**
	 * Template for comments and pingbacks.
	 * Used as a callback by wp_list_comments() for displaying the comments.
	 *
	 * @param object $comment The comment object.
	 * @param array  $args The existing args.
	 * @param int    $depth The thread depth.
	 */
	function nevo_comment( $comment, $args, $depth ) {
		global $post;
		$args['avatar_size'] = apply_filters( 'nevo_comment_avatar_size', 55 );

		if ( 'pingback' === $comment->comment_type || 'trackback' === $comment->comment_type ) : ?>

		<li id="comment-<?php comment_ID(); ?>" <?php comment_class(); ?>>
			<div class="comment-body">
				<?php esc_html_e( 'Pingback:', 'nevo' ); ?> <?php comment_author_link(); ?> <?php edit_comment_link( __( 'Edit', 'nevo' ), '<span class="edit-link">', '</span>' ); ?>
			</div>

		<?php else : ?>

		<li id="comment-<?php comment_ID(); ?>" <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ); ?>>
			<article <?php nevo_do_attr( 'comment_body', array(), array( 'comment-id' => get_comment_ID() ) ); ?>>
				<div class="comments-left d-block">
					<?php
					if ( 0 != $args['avatar_size'] ) { // phpcs:ignore
						echo get_avatar( $comment, $args['avatar_size'] );
					}
					printf(
						'<span class="comment-post-meta d-block">%1$s</span>',
						// If current post author is also comment author, make it known visually.
						( $comment->user_id === $post->post_author ) ? '' . __( 'Author', 'nevo' ) . '' : '' . __( 'Guest', 'nevo' ) . ''
					);
					/**
					 * nevo_after_comment_avatar hook.
					 *
					 * @since 1.0
					 */
					do_action( 'nevo_after_comment_avatar' );
					?>
				</div>
				<div class="comments-right cv-col-12">
					<div <?php nevo_do_attr( 'comment_meta' ); ?>>
						<div <?php nevo_do_element_classes( 'comment_author' ); ?>>
							<?php printf( '<cite itemprop="name" class="fn">%s</cite>', get_comment_author_link() ); ?>
						</div>
						<?php
						/**
						 * nevo_after_comment_author_name hook.
						 *
						 * @since 1.0
						 */
						do_action( 'nevo_after_comment_author_name' );

						if ( apply_filters( 'nevo_show_comment_entry_meta', true ) ) :
							?>
							<div class="entry-meta comment-metadata">
								<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>">
								<?php
								if ( apply_filters( 'nevo_show_comment_time_ago', true ) ) :
									?>
									<time class="d-block small" datetime="<?php comment_time( 'c' ); ?>"  itemprop="datePublished">
										<?php echo '<span class="nevo-icon" aria-hidden="true">'. nevo_get_svg_icon( 'ui', 'bi-clock-history', '', 12 ) .'</span>'; ?>
										<?php printf( esc_html__( '%s ago', 'nevo' ), human_time_diff( get_comment_time( 'U' ), current_time( 'timestamp' ) )); ?>
									</time>
								<?php else: ?>
									<time class="d-block small" datetime="<?php comment_time( 'c' ); ?>" itemprop="datePublished">
										<?php
											printf(
												/* translators: 1: date, 2: time */
												_x( '%1$s at %2$s', '1: date, 2: time', 'nevo' ), // phpcs:ignore
												get_comment_date(), // phpcs:ignore
												get_comment_time() // phpcs:ignore
											);
										?>
									</time>
								<?php endif; ?>
								</a>
							</div>
							<?php
						endif;
						?>
						<?php if ( '0' == $comment->comment_approved ) : // phpcs:ignore ?>
							<p class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'nevo' ); ?></p>
						<?php endif; ?>
					</div>

					<div class="comment-content" itemprop="text">
						<?php
						/**
						 * nevo_before_comment_content hook.
						 *
						 * @since 1.0
						 */
						do_action( 'nevo_before_comment_text', $comment, $args, $depth );

						comment_text();

						/**
						 * nevo_after_comment_content hook.
						 *
						 * @since 1.0
						 */
						?>
						<div class="d-flex flex-nowrap justify-content-between align-items-center"><div class="comment-reply">
							<?php
							do_action( 'nevo_after_comment_text', $comment, $args, $depth );
							?>
							</div>
						<?php edit_comment_link( __( 'Edit', 'nevo' ) ); ?>
						</div>
					</div>
				</div>
			</article>
			<?php
		endif;
	}
}

add_action( 'nevo_after_comment_avatar', 'nevo_comment_count', 10 );
function nevo_comment_count() {
	$comail = get_comment_author_email();
    $args = array(
		'author_email' => $comail,
		'status'  => 'approve',
		'type'  => 'comment',
	);
	$comments = get_comments($args);
	$count = count($comments);
    
    echo '<div class="nevo-comment-count small d-flex align-items-center justify-content-center" aria-hidden="true">'. nevo_get_svg_icon( 'ui', 'bi-chat', '', 10 ) .'<span class="count">' . $count .'</span></div>';
}

add_action( 'nevo_after_comment_text', 'nevo_do_comment_reply_link', 10, 3 );
/**
 * Add our comment reply link after the comment text.
 *
 * @since 1.0
 * @param object $comment The comment object.
 * @param array  $args The existing args.
 * @param int    $depth The thread depth.
 */
function nevo_do_comment_reply_link( $comment, $args, $depth ) {
	comment_reply_link(
		array_merge(
			$args,
			array(
				'add_below' => 'div-comment',
				'depth'     => $depth,
				'max_depth' => $args['max_depth'],
				'before'    => '',
				'after'     => '',
			)
		)
	);
}

add_filter( 'comment_reply_link', 'nevo_custom_comment_reply_link', 10, 4 );
function nevo_custom_comment_reply_link( $link, $args, $comment, $depth ) {
    // Your custom content
    $custom_content = '<span class="nevo-icon cv-me-2" aria-hidden="true">'. nevo_get_svg_icon( 'ui', 'bi-reply', '', 16 ) .'</span>';
    
    // Combine the custom content with the existing link
    $link = preg_replace( '/(\<a.*?\>)/', '$1' . $custom_content, $link );
    
    return $link;
}

add_filter( 'comment_form_defaults', 'nevo_set_comment_form_defaults' );
/**
 * Set the default settings for our comments.
 *
 * @since 1.0
 *
 * @param array $defaults The existing defaults.
 * @return array
 */
function nevo_set_comment_form_defaults( $defaults ) {
	$defaults['comment_field'] = sprintf(
		'<p class="comment-form-comment"><label for="comment" class="screen-reader-text">%1$s</label><textarea id="comment" name="comment" cols="45" rows="5" required></textarea></p>',
		esc_html__( 'Comment', 'nevo' )
	);

	$defaults['comment_notes_before'] = apply_filters( 'nevo_comment_notes_before', '' );
	$defaults['comment_notes_after']  = apply_filters( 'nevo_comment_notes_after', '' );
	$defaults['id_form']              = 'commentform';
	$defaults['id_submit']            = 'submit';
	$defaults['title_reply_before']   = '<h3 id="reply-title" class="comment-reply-title">';
	$defaults['title_reply_after'] 	  = '</h3>';
	$defaults['title_reply']          = apply_filters( 'nevo_leave_comment', __( 'Leave a Comment', 'nevo' ) );
	$defaults['label_submit']         = apply_filters( 'nevo_post_comment', __( 'Post Comment', 'nevo' ) );
	
	return $defaults;
}

add_filter( 'comment_form_default_fields', 'nevo_filter_comment_fields' );
/**
 * Customizes the existing comment fields.
 *
 * @since 1.0
 * @param array $fields The existing fields.
 * @return array
 */
function nevo_filter_comment_fields( $fields ) {
	$commenter = wp_get_current_commenter();
	$required = get_option( 'require_name_email' );

	$fields['author'] = sprintf(
		'<label for="author" class="screen-reader-text">%1$s</label><input placeholder="%1$s%3$s" id="author" name="author" type="text" value="%2$s" size="30"%4$s />',
		esc_html__( 'Name', 'nevo' ),
		esc_attr( $commenter['comment_author'] ),
		$required ? ' *' : '',
		$required ? ' required' : ''
	);

	$fields['email'] = sprintf(
		'<label for="email" class="screen-reader-text">%1$s</label><input placeholder="%1$s%3$s" id="email" name="email" type="email" value="%2$s" size="30"%4$s />',
		esc_html__( 'Email', 'nevo' ),
		esc_attr( $commenter['comment_author_email'] ),
		$required ? ' *' : '',
		$required ? ' required' : ''
	);

	$fields['url'] = sprintf(
		'<label for="url" class="screen-reader-text">%1$s</label><input placeholder="%1$s" id="url" name="url" type="url" value="%2$s" size="30" />',
		esc_html__( 'Website', 'nevo' ),
		esc_attr( $commenter['comment_author_url'] )
	);

	return $fields;
}

add_action( 'nevo_after_entry', 'nevo_do_comments_template', 15 );
/**
 * Add the comments template to pages and single posts.
 *
 * @since 1.0
 * @param string $template The template we're targeting.
 */
function nevo_do_comments_template() {
	$post_comments = Nevo()->get_setting( 'post_single_display_comments_element' );
	$page_comments = Nevo()->get_setting( 'page_single_display_comments_element' );
	
	if ( ( ( is_single() && $post_comments ) || ( is_page() && $page_comments ) ) && ( comments_open() || '0' != get_comments_number() ) ){
		/**
		 * nevo_before_comments_container hook.
		 *
		 * @since 1.0
		 */
		do_action( 'nevo_before_comments_container' );
		?>

		<div class="comments-area">
			<?php comments_template(); ?>
		</div>

		<?php
	}
}
// Remove field url
add_filter('comment_form_default_fields', 'nevo_remove_website_field', 9999);
function nevo_remove_website_field($fields) {
   $display_url_field = Nevo()->get_setting( 'comment_url' );

    if (! $display_url_field) {
        return $fields;
    }
    if (isset($fields['url'])) {
        unset($fields['url']);
    }
    return $fields;
}