<?php
/**
 * Abstract widget class.
 *
 * @package    Nevo
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Nevo Widget.
 *
 * Class Nevo_Widet
 *
 * @extends  WP_Widget
 */
abstract class Nevo_Widget extends WP_Widget {

	/**
	 * CSS class.
	 *
	 * @var string
	 */
	public $widget_cssclass;

	/**
	 * Widget description.
	 *
	 * @var string
	 */
	public $widget_description;

	/**
	 * Widget ID.
	 *
	 * @var string|bool
	 */
	public $widget_id = false;

	/**
	 * Widget name.
	 *
	 * @var string
	 */
	public $widget_name;

	/**
	 * Settings.
	 *
	 * @var array
	 */
	public $settings = array();

	/**
	 * Widget Control Options.
	 *
	 * @var array
	 */
	public $control_options = array();

	/**
	 * Widget customize selective refresh.
	 *
	 * @var bool
	 */
	public $customize_selective_refresh = true;

	/**
	 * Constructor.
	 */
	public function __construct() {

		$widget_options = array(
			'classname'                   => $this->widget_cssclass,
			'description'                 => $this->widget_description,
			'customize_selective_refresh' => $this->customize_selective_refresh,
		);

		parent::__construct( $this->widget_id, $this->widget_name, $widget_options, $this->control_options );

	}

	/**
	 * Updates a particular instance of a widget.
	 *
	 * @param array $new_instance New instance.
	 * @param array $old_instance Old instance.
	 *
	 * @return array
	 * @see    WP_Widget->update
	 */
	public function update( $new_instance, $old_instance ) {

		$instance = $old_instance;

		if ( empty( $this->settings ) ) {
			return $instance;
		}

		// Loop settings and get values to save.
		foreach ( $this->settings as $key => $setting ) {
			if ( ! isset( $setting['type'] ) ) {
				continue;
			}

			// Format the value based on settings type.
			switch ( $setting['type'] ) {

				case 'url':
					$instance[ $key ] = isset( $new_instance[ $key ] ) ? esc_url_raw( $new_instance[ $key ] ) : $setting['default'];
					break;

				case 'textarea':
					$instance[ $key ] = wp_kses( trim( wp_unslash( $new_instance[ $key ] ) ), wp_kses_allowed_html( 'post' ) );
					break;

				case 'image':
					/**
					 * Array of valid image file types.
					 *
					 * The array includes image mime types that are included in wp_get_mime_types()
					 */
					$mimes = array(
						'jpg|jpeg|jpe' => 'image/jpeg',
						'gif'          => 'image/gif',
						'png'          => 'image/png',
						'bmp'          => 'image/bmp',
						'tiff|tif'     => 'image/tiff',
						'ico'          => 'image/x-icon',
					);

					// Return an array with file extension and mime_type.
					$file = wp_check_filetype( $new_instance[ $key ], $mimes );

					// If $new_instance[ $key ] has a valid mime_type, assign it to $instance[ $key ], otherwise, assign empty value to $instance[ $key ].
					$instance[ $key ] = $file['ext'] ? $new_instance[ $key ] : $setting['default'];
					break;

				case 'checkbox':
					$instance[ $key ] = isset( $new_instance[ $key ] ) ? 1 : 0;
					break;

				case 'number':
					$instance[ $key ] = is_numeric( $new_instance[ $key ] ) ? intval( $new_instance[ $key ] ) : $setting['default'];

					if ( isset( $setting['input_attrs']['min'] ) && '' !== $setting['input_attrs']['min'] ) {
						$instance[ $key ] = max( $instance[ $key ], $setting['input_attrs']['min'] );
					}

					if ( isset( $setting['input_attrs']['max'] ) && '' !== $setting['input_attrs']['max'] ) {
						$instance[ $key ] = min( $instance[ $key ], $setting['input_attrs']['max'] );
					}
					break;

				case 'radio':
				case 'select':
				case 'image_select':
					$new_instance[ $key ] = sanitize_key( $new_instance[ $key ] );
					$available_choices    = $setting['choices'];

					$instance[ $key ] = array_key_exists( $new_instance[ $key ], $available_choices ) ? $new_instance[ $key ] : $setting['default'];
					break;

				case 'icons':
				case 'categories':
					$instance[ $key ] = isset( $new_instance[ $key ] ) ? $new_instance[ $key ] : array();
					break;

				case 'dropdown_categories':
					$new_instance[ $key ] = ( '-1' == $new_instance[ $key ] ) ? '0' : absint( $new_instance[ $key ] );

					$instance[ $key ] = term_exists( $new_instance[ $key ], 'category' ) ? $new_instance[ $key ] : $setting['default'];
					break;

				case 'dropdown_tags':
					$new_instance[ $key ] = absint( $new_instance[ $key ] );

					$instance[ $key ] = term_exists( $new_instance[ $key ], 'post_tag' ) ? $new_instance[ $key ] : $setting['default'];
					break;

				case 'dropdown_users':
					$new_instance[ $key ] = absint( $new_instance[ $key ] );
					$available_users      = array();
					$all_author_users     = get_users(
						array(
							'who' => 'authors',
						)
					);

					foreach ( $all_author_users as $author_user ) {
						$available_users[ $author_user->ID ] = $author_user->display_name;
					}

					$instance[ $key ] = array_key_exists( $new_instance[ $key ], $available_users ) ? $new_instance[ $key ] : $setting['default'];
					break;

				case 'checkboxes':
					$saved_data       = array();
					$instance[ $key ] = $new_instance[ $key ];

					foreach ( $instance[ $key ] as $item => $value ) {
						$saved_data[ $item ] = isset( $item ) ? 1 : 0;
					}

					$instance[ $key ] = $saved_data;
					break;

				case 'numbers':
					$saved_data       = array();
					$instance[ $key ] = $new_instance[ $key ];

					foreach ( $instance[ $key ] as $item => $value ) {
						$temp_data = is_numeric( $value ) ? intval( $value ) : $setting['default'][ $item ];

						if ( isset( $setting['input_attrs']['min'] ) && '' !== $setting['input_attrs']['min'] && ( $value < $setting['input_attrs']['min'] && ! $temp_data ) ) {
							$temp_data = max( $value, $setting['input_attrs']['min'] );
						}

						if ( isset( $setting['input_attrs']['max'] ) && '' !== $setting['input_attrs']['max'] && $value > $setting['input_attrs']['max'] ) {
							$temp_data = min( $value, $setting['input_attrs']['max'] );
						}

						$saved_data[ $item ] = $temp_data;
					}

					$instance[ $key ] = $saved_data;
					break;

				case 'multiselect':
					$selected_choices     = array();
					$available_choices    = $setting['choices'];
					$new_instance[ $key ] = isset( $new_instance[ $key ] ) ? $new_instance[ $key ] : array();

					foreach ( $new_instance[ $key ] as $selected_key => $selected_value ) {

						if ( array_key_exists( $selected_value, $available_choices ) ) {
							$selected_choices[] = $selected_value;
						}
					}

					$instance[ $key ] = $selected_choices;
					break;

				default:
					$instance[ $key ] = isset( $new_instance[ $key ] ) ? sanitize_text_field( $new_instance[ $key ] ) : $setting['default'];
					break;

			}

			/**
			 * Sanitize the value of a setting.
			 */
			$instance[ $key ] = apply_filters( 'nevo_widget_settings_sanitize_option', $instance[ $key ], $new_instance, $key, $setting );
		}

		return $instance;

	}

	/**
	 * Outputs the settings update form.
	 *
	 * @param array $instance Instance.
	 *
	 * @see   WP_Widget->form
	 */
	public function form( $instance ) {

		if ( empty( $this->settings ) ) {
			return;
		}
		
		foreach ( $this->settings as $key => $setting ) {

			$class = isset( $setting['class'] ) ? $setting['class'] : '';
			$value = isset( $instance[ $key ] ) ? $instance[ $key ] : $setting['default'];
			$category = isset( $instance[ $key ] ) ? $instance[ $key ] : array();
			
			echo '<div class="cv-col-content '. esc_attr( isset( $setting['col'] ) ? $setting['col'] : 'col-0' ) .'">';
			switch ( $setting['type'] ) {
			
				case 'text':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<input type="text"
						       class="widefat <?php echo esc_attr( $class ); ?>"
						       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						       value="<?php echo esc_attr( $value ); ?>"
						/>
					</p>
					<?php
					break;

				case 'url':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<input type="url"
						       class="widefat <?php echo esc_attr( $class ); ?>"
						       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						       value="<?php echo esc_attr( $value ); ?>"
						/>
					</p>
					<?php
					break;

				case 'textarea':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<textarea class="widefat <?php echo esc_attr( $class ); ?>"
						          rows="5"
						          cols="20"
						          id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						          name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						><?php echo esc_textarea( $value ); ?></textarea>
					</p>
					<?php
					break;

				case 'image':
					?>
					<div class="media-uploader">
						<p><label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label></p>
						<div class="custom_media_preview">
							<?php if ( $value ) : ?>
								<img class="custom_media_preview_default"
									 src="<?php echo esc_url( $value ); ?>"
									 style="max-width:100%;"
								/>
							<?php endif; ?>
						</div>

						<input type="text"
							   class="widefat custom_media_input"
							   id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
							   name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
							   value="<?php echo esc_attr( $value ); ?>"
							   style="margin-top:5px;"
						/>

						<button class="custom_media_upload button button-secondary button-large"
								id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
								data-choose="<?php esc_attr_e( 'Choose an image', 'nevo' ); ?>"
								data-update="<?php esc_attr_e( 'Use image', 'nevo' ); ?>"
								style="width:100%;margin-top:6px;margin-bottom:30px;"
						>
							<?php esc_html_e( 'Select an Image', 'nevo' ); ?>
						</button>
					</div>
					<?php
					break;

				case 'checkbox':
					?>
					<p>
						<input class="checkbox"
						       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						       type="checkbox"
							<?php echo esc_attr( ( $value == 1 ) ? 'checked' : '' ); ?>
						/>

						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
					</p>
					<?php
					break;

				case 'number':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<input class="widefat <?php echo esc_attr( $class ); ?>"
						       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						       type="number"
						       value="<?php echo esc_attr( $value ); ?>"
							<?php if ( isset( $setting['input_attrs']['step'] ) ) { ?>
								step="<?php echo esc_attr( $setting['input_attrs']['step'] ); ?>"
							<?php } ?>
							<?php if ( isset( $setting['input_attrs']['min'] ) ) { ?>
								min="<?php echo esc_attr( $setting['input_attrs']['min'] ); ?>"
							<?php } ?>
							<?php if ( isset( $setting['input_attrs']['max'] ) ) { ?>
								max="<?php echo esc_attr( $setting['input_attrs']['max'] ); ?>"
							<?php } ?>
						/>
					</p>
					<?php
					break;

				case 'radio':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<?php
						$count = 1;
						foreach ( $setting['choices'] as $choices_key => $choices_value ) {
							if ( 1 !== $count ) {
								echo '<br />';
							}
							?>
							<input type="radio"
							   id="<?php echo esc_attr( $this->get_field_id( $choices_key ) ); ?>"
							   name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
							   value="<?php echo esc_attr( $choices_key ); ?>"
							<?php echo esc_attr( ( $choices_key == $value ) ? 'checked' : '' ); ?>
							/>
							<label for="<?php echo esc_attr( $this->get_field_id( $choices_key ) ); ?>">
								<?php echo esc_html( $choices_value ); ?>
							</label>
							<?php
							$count ++;
						}
						?>
					</p>
					<?php
					break;
					
				case 'image_select':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<?php
						echo '<div class="cv-radio-images">';
							foreach ( $setting['choices'] as $choices_key => $choices_value ) {
								?>
								<p>
									<input type="radio"
										   id="<?php echo esc_attr( $this->get_field_id( $choices_key ) ); ?>"
										   name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
										   value="<?php echo esc_attr( $choices_key ); ?>"
										<?php echo esc_attr( ( $choices_key == $value ) ? 'checked' : '' ); ?>
									/>
									<label for="<?php echo esc_attr( $this->get_field_id( $choices_key ) ); ?>">
											<img src="<?php echo esc_url( $choices_value ); ?>" alt="" />
									</label>
								</p>
								<?php
							}
						echo '</div>';
						?>
					</p>
					<?php
					break;

				case 'select':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<select class="widefat <?php echo esc_attr( $class ); ?>"
						        id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"
						        name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>"
						>
							<?php foreach ( $setting['choices'] as $choices_key => $choices_value ) { ?>
								<option value="<?php echo esc_attr( $choices_key ); ?>"
									<?php selected( $choices_key, $value ); ?>
								>
									<?php echo esc_html( $choices_value ); ?>
								</option>
							<?php } ?>
						</select>
					</p>
					<?php
					break;

				case 'custom':
					?>
					<div class="custom">
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>">
							<?php
							echo $setting['label']; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
							?>
						</label>

						<?php if ( isset( $setting['image_url'] ) ) { ?>
							<div style="text-align: center;">
								<img src="<?php echo esc_url( $setting['image_url'] ); ?>" alt="" />
							</div>
						<?php } ?>
					</div>
					<?php
					break;
					
				case 'categories':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
						<?php
						$value_terms = isset( $setting['taxonomy'] ) ? $setting['taxonomy'] : 'category';
						$taxonomies  = get_terms( array(
							'taxonomy'   => $value_terms,
							'orderby'    => 'name',
							'order'      => 'ASC',
							'hide_empty' => false
						) );
						if ( ! is_wp_error( $taxonomies ) && ! empty( $taxonomies ) ) :
							$category_tree = nevo_build_category_tree( $taxonomies, 0 );
							?>
							<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>[]" multiple>
								<?php nevo_display_category_tree( $category_tree, $category ); ?>
							</select>
						<?php endif; ?>
					</p>
					<?php
					break;

				case 'dropdown_categories':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
						<?php
						$dropdown_args = array(
							'show_option_all'  => esc_html__('Select Category','nevo'),
							'name'             => $this->get_field_name( $key ),
							'selected'         => $value,
							'class'            => 'widefat postform',
							'hierarchical' 	   => 1,
						);
						if ( isset( $setting['taxonomy'] ) ) {
							$dropdown_args['taxonomy'] = $setting['taxonomy']; //'product_cat';
						}
						if ( isset( $setting['parent'] ) ) {
							$dropdown_args['parent'] = '0';
						}
						wp_dropdown_categories( $dropdown_args );
						?>
					</p>
					<?php
					break;

				case 'dropdown_tags':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
						<?php
						wp_dropdown_categories(
							array(
								'show_option_all'  => esc_html__('Select Tag','nevo'),
								'name'             => $this->get_field_name( $key ),
								'selected'         => $value,
								'taxonomy'         => 'post_tag',
								'class'            => 'widefat postform',
							)
						);
						?>
					</p>
					<?php
					break;

				case 'dropdown_users':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>
						<?php
						wp_dropdown_users(
							array(
								'show_option_none' => ' ',
								'name'             => $this->get_field_name( $key ),
								'selected'         => $value,
								'orderby'          => 'name',
								'order'            => 'ASC',
								'who'              => 'authors',
								'class'            => 'widefat postform',
							)
						);
						?>
					</p>
					<?php
					break;

				case 'separator':
					?>
					<hr />
					<?php
					break;

				case 'checkboxes':
					?>
					<h3><?php echo esc_html( $setting['label'] ); ?></h3>

					<p>
						<?php foreach ( $setting['choices'] as $choices_key => $choices_value ) { ?>
							<label class="alignleft"
							       style="width:50%;display:block;margin-bottom:5px"
							       for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>_<?php echo esc_attr( $choices_key ); ?>"
							>

								<input type="checkbox"
								       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>_<?php echo esc_attr( $choices_key ); ?>"
								       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>[<?php echo esc_attr( $choices_key ); ?>]"
								       value="1"
									<?php
									if ( isset( $value[ $choices_key ] ) ) {
										checked( 1, $value[ $choices_key ], true );
									}
									?>
								/>

								<?php echo esc_html( $choices_value ); ?>
							</label>
						<?php } ?>
					</p>

					<div class="clear"></div>
					<?php
					break;

				case 'numbers':
					?>
					<h3><?php echo esc_html( $setting['label'] ); ?></h3>

					<p>
						<?php foreach ( $setting['choices'] as $choices_key => $choices_value ) { ?>
							<label class="alignleft"
							       style="width:50%;display:block;margin-bottom:5px"
							       for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>_<?php echo esc_attr( $choices_key ); ?>"
							>

								<input type="number"
								       id="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>_<?php echo esc_attr( $choices_key ); ?>"
								       name="<?php echo esc_attr( $this->get_field_name( $key ) ); ?>[<?php echo esc_attr( $choices_key ); ?>]"
								       value="<?php echo esc_attr( $value[ $choices_key ] ); ?>"
									<?php if ( isset( $setting['input_attrs']['step'] ) ) { ?>
										step="<?php echo esc_attr( $setting['input_attrs']['step'] ); ?>"
									<?php } ?>
									<?php if ( isset( $setting['input_attrs']['min'] ) ) { ?>
										min="<?php echo esc_attr( $setting['input_attrs']['min'] ); ?>"
									<?php } ?>
									<?php if ( isset( $setting['input_attrs']['max'] ) ) { ?>
										max="<?php echo esc_attr( $setting['input_attrs']['max'] ); ?>"
									<?php } ?>
								/>

								<?php echo esc_html( $choices_value ); ?>
							</label>
						<?php } ?>
					</p>

					<div class="clear"></div>
					<?php
					break;

				case 'api_key':
					$api_key = $setting['api_key'];

					if ( ! $api_key ) {
						$query['autofocus[control]'] = $setting['customize_control'];
						$control_link                = add_query_arg( $query, admin_url( 'customize.php' ) );
						?>
						<p>
							<span class="<?php echo esc_attr( $setting['class'] ); ?>-error">
								<?php echo $setting['description']; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped ?>
								<br />
								<a href="<?php echo esc_url( $control_link ); ?>"><?php echo esc_html( $setting['label'] ); ?></a>
							</span>
						</p>
						<?php
					}
					break;

				case 'multiselect':
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( $key ) ); ?>"><?php echo esc_html( $setting['label'] ); ?></label>

						<?php
						printf(
						/* Translators: 1. Field name, 2. Field id, 3. Custom style declaration */
							'<select multiple="multiple" name="%s[]" id="%s" %s>',
							esc_attr( $this->get_field_name( $key ) ),
							esc_attr( $this->get_field_id( $key ) ),
							'style="width:100%"'
						);

						$available_values = ! empty( $value ) ? $value : array();

						foreach ( $setting['choices'] as $choices_key => $choices_value ) {
							?>
							<option value="<?php echo esc_attr( $choices_key ); ?>"
								<?php
								if ( in_array( $choices_key, $available_values, true ) ) {
									echo ' selected="selected"';
								}
								?>
							>
								<?php echo esc_html( $choices_value ); ?>
							</option>
							<?php
						}

						echo '</select>';
						?>

					</p>
					<?php
					break;
					
				// Default: run an action.
				default:
					do_action( 'nevo_widget_field_' . $setting['type'], $key, $value, $setting, $instance );
					break;
			}
			echo '</div>';
		}
	}

	/**
	 * Output the html at the start of a widget.
	 *
	 * @param array $args Arguments.
	 */
	public function widget_start( $args ) {
		echo $args['before_widget']; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Output the html at the end of a widget.
	 *
	 * @param array $args Arguments.
	 */
	public function widget_end( $args ) {
		echo $args['after_widget']; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Displays the widget title within the widgets.
	 *
	 * @param string $title    The widget title.
	 * @param string $type     The display posts from the widget setting.
	 * @param int    $category The category id of the widget setting.
	 */
	public function widget_title( $title, $type, $category ) {

		// Return if $title is empty.
		if ( ! $title ) {
			return;
		}

		$border_color   = '';
		$title_color    = '';
		$category_color = nevo_category_color( $category );
		if ( 'latest' != $type && $category_color ) {
			$border_color = 'style="border-bottom-color:' . $category_color . ';"';
			$title_color  = 'style="background-color:' . $category_color . ';"';
		}

		// Display the title.
		if ( ! empty( $title ) ) {
			echo '<div class="heading-title" ' . $border_color . '><h2 class="widget-title general-title" ' . $title_color . '>' . esc_html( $title ) . '</h2></div>'; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
		}

	}

	/**
	 * Displays the widget description within the widgets.
	 *
	 * @param string $text The widget description.
	 */
	public function widget_description( $text ) {

		// Return if $text is empty.
		if ( ! $text ) {
			return;
		}

		echo '<p>' . wp_kses_post( $text ) . '</p>';

	}
	
	/**
	 * Query of the posts within the widgets.
	 *
	 * @param int    $number   The number of posts to display.
	 * @param int    $orderby  The Sort By of posts to display.
	 * @param string $all_categories     The display posts from the widget setting.
	 * @param int    $category The category id of the widget setting.
	 *
	 * @return \WP_Query
	 */
	public function query_posts( $number, $all_categories, $category, $orderby ) {

		$post_status = 'publish';
		if ( 1 == get_option( 'fresh_site' ) ) {
			$post_status = array( 'auto-draft', 'publish' );
		}

		$query_args = array(
			'posts_per_page'      => $number,
			'post_type'           => 'post',
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
			'orderby'             => $orderby,
			'post_status'         => $post_status,
		);

		// Display posts from category.
		if ( ! $all_categories ) $query_args['cat'] = implode( ',', $category );

		$query = new WP_Query( $query_args );

		return $query;

	}

}