<?php
/**
 * WooCommerce Template Functions.
 *
 * @package nevo
 */

if ( ! function_exists( 'nevo_output_content_wrapper_start' ) ) {
	/**
	 * Add WooCommerce starting wrappers
	 *
	 * @since 1.0
	 */
	function nevo_output_content_wrapper_start() {
		?>
		<div <?php nevo_do_attr( 'content' ); ?>>
			<main <?php nevo_do_attr( 'main' ); ?>>
				<?php
				/**
				 * nevo_before_while hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_before_while' );
				?>
				<div class="product-content">
				<?php
				/**
				 * nevo_before_product_content hook.
				 *
				 * @since 0.1
				 */
				do_action( 'nevo_before_product_content' );
				
	}
}

if ( ! function_exists( 'nevo_output_content_wrapper_end' ) ) {
	/**
	 * Add WooCommerce ending wrappers
	 *
	 * @since 1.0
	 */
	function nevo_output_content_wrapper_end() {
				
				/**
				 * nevo_after_product_content hook.
				 *
				 * @since 0.1
				 */
				do_action( 'nevo_after_product_content' );
				?>
				</div>
				<?php
				/**
				 * nevo_after_endwhile hook.
				 *
				 * @since 1.0
				 */
				do_action( 'nevo_after_endwhile' );
				?>
			</main>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_shop_messages' ) ) {
	/**
	 * Nevo shop messages
	 *
	 * @since   1.0
	 * @uses    nevo_do_shortcode
	 */
	function nevo_shop_messages() {
		if ( ! is_checkout() && is_woocommerce() ) {
			echo wp_kses_post( nevo_do_shortcode( 'woocommerce_messages' ) );
		}
	}
}

if ( ! function_exists( 'nevo_woo_cart_available' ) ) {
	/**
	 * Validates whether the Woo Cart instance is available in the request
	 *
	 * @since 1.0
	 * @return bool
	 */
	function nevo_woo_cart_available() {
		$woo = WC();
		return $woo instanceof \WooCommerce && $woo->cart instanceof \WC_Cart;
	}
}

/**
 * ========================================================================================================
 * Archive Products Start
 * ========================================================================================================
 */
if ( ! function_exists( 'nevo_sorting_wrapper' ) ) {
	/**
	 * Sorting wrapper
	 */
	function nevo_sorting_wrapper() {
		echo '<div class="nevo-sorting d-flex align-items-center">';
	}
}

if ( ! function_exists( 'nevo_sorting_wrapper_close' ) ) {
	/**
	 * Sorting wrapper close
	 */
	function nevo_sorting_wrapper_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'nevo_woocommerce_loop_start' ) ) {
	/**
	 * Modify: Loop start
	 *
	 * @param string $loop_start The loop start.
	 */
	function nevo_woocommerce_loop_start( $loop_start ) {
		$mobile = intval( Nevo()->get_setting( 'product_archive_product_col', 'mobile' ) );
		$tablet = intval( Nevo()->get_setting( 'product_archive_product_col', 'tablet' ) );
		$columns = wc_get_loop_prop( 'columns' );
		
		$per_row_class[] = 'products';
		$per_row_class[] = 'list-inline';
		$per_row_class[] = 'cv-row';
		if ( $mobile <= $columns && $columns >= 1 ) {
			$per_row_class[] = 'row-cols-'. $mobile;
		} else {
			$per_row_class[] = 'row-cols-'. $columns;
		}
		if ( $tablet <= $columns && $columns > 1 ) {
			$per_row_class[] = 'row-cols-sm-'. $tablet;
		}
		if ( $columns > 1 ) {
			$per_row_class[] = 'row-cols-lg-'. $columns;
		}
		$class   = implode( ' ', $per_row_class );
		?>
		<ul class="<?php echo esc_attr( apply_filters( 'nevo_product_catalog_columns', $class ) ); ?>">
		<?php

		// If displaying categories, append to the loop.
		$loop_html = woocommerce_maybe_show_product_subcategories();
		echo $loop_html; // phpcs:ignore
	}
}

if ( ! function_exists( 'nevo_template_loop_product_inner_open' ) ) {
	/**
	 * Col product wrapper open tag
	 */
	function nevo_template_loop_product_inner_open() {
		?>
		<div class="col-inner position-relative h-100">
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_category_title_override' ) ) {
	function nevo_template_loop_category_title_override( $category ) { ?>
		<div class="box-text">
			<h2 class="woocommerce-loop-category__title">
				<?php echo esc_html( $category->name ); ?>
			</h2><?php
			if ( $category->count > 0 ) {
				echo apply_filters( 'woocommerce_subcategory_count_html', ' <p class="small count">' . $category->count . ' ' . ( $category->count > 1 ? __( 'Products', 'nevo' ) : __( 'Product', 'nevo' ) ) . '</p>', $category );
			} ?>
		</div><?php
	}
}

if ( ! function_exists( 'nevo_template_loop_product_badge_wrap_open' ) ) {
	/**
	 * Badge product wrapper open tag
	 */
	function nevo_template_loop_product_badge_wrap_open() {
		?>
		<div class="badge-wrap badge position-absolute">
		<?php
	}
}

if ( ! function_exists( 'nevo_change_sale_flash' ) ) {
	/**
	 * Change sale flash
	 */
	function nevo_change_sale_flash() {
		global $product;
		if ( ! $product || ! is_object( $product ) ) {
			return;
		}
		$sale         = $product->is_on_sale();
		$price_sale   = $product->get_sale_price();
		$price        = $product->get_regular_price();
		$simple       = $product->is_type( 'simple' );
		$variable     = $product->is_type( 'variable' );
		$external     = $product->is_type( 'external' );
		$bundle       = $product->is_type( 'bundle' );
		$sale_text    = Nevo()->get_setting( 'product_archive_sale_text' ); //text
		$sale_percent = Nevo()->get_setting( 'product_archive_sale_percentage' ); // có hay không quy đổi %
		$final_price  = '';
		$out_of_stock = get_post_meta( $product->get_id(), '_stock_status', true );

		if ( 'outofstock' === $out_of_stock ) {
			return;
		}

		if ( $sale ) {
			// For simple product.
			if ( $simple || $external || $bundle ) {
				if ( $sale_percent ) {
					$final_price = ( ( $price - $price_sale ) / $price ) * 100;
					$final_price = '-' . round( $final_price ) . '%';
				} elseif ( $sale_text ) {
					$final_price = $sale_text;
				}
			} elseif ( $variable && $sale_text ) {
				// For variable product.
				$final_price = $sale_text;
			}

			if ( ! $final_price ) {
				return;
			}

			$final_price = apply_filters( 'nevo_price_flash', $final_price, $product );

			$classes[] = 'on-sale onsale has-success-background-color';
			?>
			<span class="<?php echo esc_attr( implode( ' ', array_filter( $classes ) ) ); ?>">
				<?php echo esc_html( $final_price ); ?>
			</span>
			<?php
		}
	}
}

if ( ! function_exists( 'nevo_print_featured_label' ) ) {
	/**
	 * print featured label
	 */
	function nevo_print_featured_label() {
		global $product;
		if ( ! $product ) {
			return;
		}
		$product_id    = $product->get_id();
		$featured	   = $product->is_featured();
		$featured_text = Nevo()->get_setting( 'product_archive_featured_text' );
		$featuredtext  = Nevo()->get_setting( 'product_archive_show_featured' );
		$out_of_stock  = get_post_meta( $product_id, '_stock_status', true );
		$classes[] = 'product-labels';

		// Out of stock.
		if ( 'outofstock' === $out_of_stock ) {
			return;
		}
		/* Hot label */
		if ( $featuredtext && $featured ) { ?>
			<span class="callout featured-label bg-warning"><?php echo esc_html( $featured_text ); ?></span><?php
		}
	}
}

if ( ! function_exists( 'nevo_template_loop_product_badge_wrap_close' ) ) {
	/**
	 * Badge product wrapper open tag
	 */
	function nevo_template_loop_product_badge_wrap_close() {
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_product_image_wrapper_open' ) ) {
	/**
	 * Image product wrapper open tag
	 */
	function nevo_template_loop_product_image_wrapper_open() {
		$class[] = 'product-image-wrap';
		$class[] = 'text-center';
		$class[] = 'position-relative';
		if ( intval( Nevo()->get_setting( 'product_archive_equalimg_height' ) ) ) {
			$class[] = 'equal-height';
		}
		if ( 'zoom' == Nevo()->get_setting( 'product_archive_img_hover' ) ) {
			$class[] = 'image-zoom';
		}
		$class   = implode( ' ', $class );
		?>
		<div class="<?php echo esc_attr( apply_filters( 'nevo_template_loop_product_image_wrapper_class', $class ) ); ?>">
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_product_thumbnail_hover' ) ) {
	/**
	 * Loop product hover image
	 */
	function nevo_template_loop_product_thumbnail_hover() {
		if ( 'swap' !== Nevo()->get_setting( 'product_archive_img_hover' ) ) {
			return;
		}
		global $product;
		$gallery    = $product->get_gallery_image_ids();
		$size       = 'woocommerce_thumbnail';
		$image_size = apply_filters( 'single_product_archive_thumbnail_size', $size );

		// Hover image.
		if ( ! empty( $gallery ) ) {
			$hover = wp_get_attachment_image_src( $gallery[0], $image_size );
			if ( ! empty( $hover ) ) {
				?>
				<span class="product-loop-hover-image" style="background-image: url(<?php echo esc_url( $hover[0] ); ?>);"></span>
				<?php
			}
		}
	}
}

if ( ! function_exists( 'nevo_template_loop_product_thumbnail' ) ) {
	/**
	 * Loop product image
	 */
	function nevo_template_loop_product_thumbnail() {
		global $product;

		if ( ! $product ) {
			return '';
		}

		$size    = 'woocommerce_thumbnail';
		$img_id  = $product->get_image_id();
		$img_alt = nevo_image_alt( $img_id, esc_attr__( 'Product image', 'nevo' ) );

		$image_attr = array(
			'alt'   => $img_alt,
			'class' => 'attachment-' . $size . ' size-' . $size . ' product-loop-image',
		);

		echo $product->get_image( $size, $image_attr ); // phpcs:ignore
	}
}

if ( ! function_exists( 'nevo_product_image_tool_wrapper_open' ) ) {
	/**
	 * Tool product wrapper open tag
	 */
	function nevo_product_image_tool_wrapper_open() {
		?>
		<div class="image-tools position-absolute">
		<?php
	}
}

if ( ! function_exists( 'nevo_product_image_tool_wrapper_close' ) ) {
	/**
	 * Tool product wrapper close tag
	 */
	function nevo_product_image_tool_wrapper_close() {
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_print_out_of_stock_label' ) ) {
	/**
	 * Print out of stock label
	 */
	function nevo_print_out_of_stock_label() {
		global $product;

		if ( ! $product ) {
			return;
		}

		$product_id   = $product->get_id();
		$out_of_stock = get_post_meta( $product_id, '_stock_status', true );
		$product_type = WC_Product_Factory::get_product_type( $product_id );
		$stocklabel = Nevo()->get_setting( 'product_archive_stock_text' );

		if ( ! $out_of_stock || 'external' === $product_type || $product->backorders_allowed() ) {
			return;
		}

		if ( 'outofstock' === $out_of_stock && $stocklabel ) {
			?>
			<span class="out-of-stock-label"><?php echo esc_html( $stocklabel ); ?></span>
			<?php
		}
	}
}

if ( ! function_exists( 'nevo_template_loop_product_image_wrapper_close' ) ) {
	/**
	 * Image product wrapper close tag
	 */
	function nevo_template_loop_product_image_wrapper_close() {
		?>
		</div>
		<?php
	}
}
	
if ( ! function_exists( 'nevo_template_loop_product_content_wrap_open' ) ) {
	/**
	 * Content product wrapper open tag
	 */
	function nevo_template_loop_product_content_wrap_open() {
		?>
		<div class="product-content-wrap">
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_product_item' ) ) {
	function nevo_template_loop_product_item() {
		$items = apply_filters(
			'nevo_product_loop_items',
			array( 'category', 'title', 'price', 'description', 'rating', 'quantity' ) // , 'rating', 'description', 'quantity'
		);
		if ( empty( $items ) ) {
			return;
		}
		foreach ( $items as $item ) {
			if ( intval( Nevo()->get_setting( "product_archive_items_{$item}" ) ) ) {
				nevo_do_product_loop_item( $item );
			}
		}
	}
}

function nevo_do_product_loop_item( $item ) {
	global $woocommerce, $product;
	if ( 'category' === $item ) {
		echo '<div class="woocommerce-loop-category__title">';
			$product_id = $product->get_ID();
			echo wp_kses_post( wc_get_product_category_list( $product_id ) );
		echo '</div>';
	}
	if ( 'title' === $item ) {
		echo '<h2 class="woocommerce-loop-product__title">';
			woocommerce_template_loop_product_link_open();
			the_title();
			woocommerce_template_loop_product_link_close();
		echo '</h2>';
	}
	if ( 'price' === $item ) {
		$price_html = $product->get_price_html();
		if ( $price_html ) {
			echo '<div class="price">' . wp_kses_post( $price_html ) . '</div>';
		}
	}
	if ( 'description' === $item ) {
		if ( has_excerpt() ) {
			echo '<div class="product-description">';
				the_excerpt();
			echo '</div>';
		}
	}
	if ( 'rating' === $item ) {
		$review_count = $product->get_review_count();
		$average = $product->get_average_rating();
		
		$html = '<div class="woocommerce-product-rating">';
		$html .= '<div class="star-rating" aria-label="Rated '.$average.' out of 5"><span style="width:'.( ( $average / 5 ) * 100 ) . '%"><strong itemprop="ratingValue" class="rating">'.$average.'</strong> '.__( 'out of 5', 'nevo' ).'</span></div>';
		$html .= '<div class="count">(' . esc_html( $review_count ) . ')</div>';
		$html .= '</div>';
		
		echo $html; // phpcs:ignore.
	}
	if ( 'quantity' === $item ) {
		$product = wc_get_product( get_the_ID() );

		if ( $product->is_sold_individually() || 'variable' === $product->get_type() || ! $product->is_purchasable() ) {
			return;
		}

		$html = '';

		$html .= '<div class="loop-product-qty">';
		$html .= woocommerce_quantity_input(
			array(
				'min_value' => 1,
				'max_value' => $product->backorders_allowed() ? '' : $product->get_stock_quantity(),
			),
			$product,
			false
		);
		$html .= '</div>';

		echo $html; // phpcs:ignore.
	}
	/**
	 * nevo_product_loop_item hook.
	 *
	 * @since 1.0
	 */
	do_action( 'nevo_product_loop_item', $item );
}

if ( ! function_exists( 'nevo_template_loop_add_to_cart' ) ) {
	/**
	 * add to cart button
	 */
	function nevo_template_loop_add_to_cart() {
		$args = nevo_modify_loop_add_to_cart();
		echo '<div class="add-links-wrap">';
			woocommerce_template_loop_add_to_cart( $args );
		echo '</div>';
	}
}

if ( ! function_exists( 'nevo_modify_loop_add_to_cart' ) ) {
	/**
	 * Modify loop add to cart
	 */
	function nevo_modify_loop_add_to_cart() {
		global $product;
		$button_class = 'loop-add-to-cart-btn';
		$icon_class   = '';//'nevo-popup-toggle';

		$args = array(
			'class'      => implode(
				' ',
				array_filter(
					array(
						$icon_class,
						$button_class,
						'button',
						'product_type_' . $product->get_type(),
						$product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '',
						$product->supports( 'ajax_add_to_cart' ) && $product->is_purchasable() && $product->is_in_stock() ? 'ajax_add_to_cart' : '',
					)
				)
			),
			'attributes' => array(
				//'data-target'      => 'header-mini-cart-vertical',
				'data-product_id'  => $product->get_id(),
				'data-product_sku' => $product->get_sku(),
				'title'            => $product->add_to_cart_description(),
				'rel'              => 'nofollow',
			),
		);

		return $args;
	}
}

if ( ! function_exists( 'nevo_template_loop_product_content_wrap_close' ) ) {
	/**
	 * Content product wrapper close tag
	 */
	function nevo_template_loop_product_content_wrap_close() {
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_product_inner_close' ) ) {
	/**
	 * Col product wrapper close tag
	 */
	function nevo_template_loop_product_inner_close() {
		?>
		</div>
		<?php
	}
}
 
/**
 * ========================================================================================================
 * Single Products Start
 * ========================================================================================================
 */

if ( ! function_exists( 'nevo_template_loop_single_product_entry_wrap_open' ) ) {
	/**
	 * Main wrapper
	 */
	function nevo_template_loop_single_product_entry_wrap_open() {
		?>
		<div class="product-main-wrap d-md-flex">
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_single_product_images_wrap_open' ) ) {
	/**
	 * Images wrapper
	 */
	function nevo_template_loop_single_product_images_wrap_open() {
		?>
		<div class="product-images-wrap position-relative">
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_single_product_images_wrap_close' ) ) {
	/**
	 * Images wrapper close
	 */
	function nevo_template_loop_single_product_images_wrap_close() {
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_template_loop_single_product_entry_wrap_close' ) ) {
	/**
	 * Main wrapper close
	 */
	function nevo_template_loop_single_product_entry_wrap_close() {
		?>
		</div>
		<?php
	}
}

/////////////////

if ( ! function_exists( 'nevo_template_loop_single_product_footer_wrap_open' ) ) {
	/**
	 * Images wrapper
	 */
	function nevo_template_loop_single_product_footer_wrap_open() {
		?>
		<div class="product-footer-wrap">
		<?php
	}
}

if ( ! function_exists( 'nevo_output_product_data_tabs' ) ) {
	/**
	 * Filter tabs and allow third parties to add their own.
	 *
	 * Each tab is an array containing title, callback and priority.
	 *
	 * @see woocommerce_default_product_tabs()
	 */
	function nevo_output_product_data_tabs() {
		$product_tabs = apply_filters( 'woocommerce_product_tabs', array() );
		if ( ! empty( $product_tabs ) ) :
			$tab_type = Nevo()->get_setting( 'product_single_tabs' );
			if ( 'normal' === $tab_type ) { ?>
				<div class="woocommerce-tabs wc-tabs-wrapper">
					<ul class="tabs wc-tabs d-flex tabs-list" role="tablist">
						<?php foreach ( $product_tabs as $key => $product_tab ) : ?>
							<li class="tab-title <?php echo esc_attr( $key ); ?>_tab" id="tab-title-<?php echo esc_attr( $key ); ?>" role="tab" aria-controls="tab-<?php echo esc_attr( $key ); ?>">
								<a class="d-block position-relative" href="#tab-<?php echo esc_attr( $key ); ?>">
									<?php echo wp_kses_post( apply_filters( 'woocommerce_product_' . $key . '_tab_title', $product_tab['title'], $key ) ); ?>
								</a>
							</li>
						<?php endforeach; ?>
					</ul>
					<?php foreach ( $product_tabs as $key => $product_tab ) : ?>
						<div class="woocommerce-Tabs-panel woocommerce-Tabs-panel--<?php echo esc_attr( $key ); ?> panel tabs-content wc-tab" id="tab-<?php echo esc_attr( $key ); ?>" role="tabpanel" aria-labelledby="tab-title-<?php echo esc_attr( $key ); ?>">
							<?php
							if ( isset( $product_tab['callback'] ) ) {
								call_user_func( $product_tab['callback'], $key, $product_tab );
							}
							?>
						</div>
					<?php endforeach; ?>

					<?php do_action( 'woocommerce_product_after_tabs' ); ?>
				</div>
			<?php }else{ ?>
				<div class="woocommerce-tabs wc-tabs-wrapper">
					<ul class="tabs wc-tabs tabs-accordion" role="tablist">
						<?php foreach ( $product_tabs as $key => $product_tab ) : ?>
							<div class="accordion--item d-block">
								<li class="tab-title <?php echo esc_attr( $key ); ?>_tab" id="tab-title-<?php echo esc_attr( $key ); ?>" role="tab" aria-controls="tab-<?php echo esc_attr( $key ); ?>">
									<a class="d-block" href="#tab-<?php echo esc_attr( $key ); ?>">
										<?php echo wp_kses_post( apply_filters( 'woocommerce_product_' . $key . '_tab_title', $product_tab['title'], $key ) ); ?>
									</a>
								</li>
								<div class="woocommerce-Tabs-panel woocommerce-Tabs-panel--<?php echo esc_attr( $key ); ?> panel tabs-content wc-tab" id="tab-<?php echo esc_attr( $key ); ?>" role="tabpanel" aria-labelledby="tab-title-<?php echo esc_attr( $key ); ?>">
									<?php
									if ( isset( $product_tab['callback'] ) ) {
										call_user_func( $product_tab['callback'], $key, $product_tab );
									}
									?>
								</div>
							</div>
						<?php endforeach; ?>
					</ul>
					<?php do_action( 'woocommerce_product_after_tabs' ); ?>
				</div> <?php
			}
		endif;
	}
}

if ( ! function_exists( 'nevo_template_loop_single_product_footer_wrap_close' ) ) {
	/**
	 * Images wrapper close
	 */
	function nevo_template_loop_single_product_footer_wrap_close() {
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_upsell_display' ) ) {
	/**
	 * Upsells
	 * Replace the default upsell function with our own which displays the correct number product columns
	 *
	 * @since   1.0.0
	 * @return  void
	 * @uses    woocommerce_upsell_display()
	 */
	function nevo_upsell_display() {
		$columns = intval( Nevo()->get_setting( 'product_single_upsells_col' ) );
		woocommerce_upsell_display( -1, $columns );
	}
}

/**
 * ========================================================================================================
 * Products Cart Page
 * ========================================================================================================
 */

if ( ! function_exists( 'nevo_cross_sell_display_columns' ) ) {
	/**
	 * Cross sell product columns
	 *
	 * @param string $columns.
	 */
	function nevo_cross_sell_display_columns( $columns ) {
		$columns = intval( Nevo()->get_setting( 'product_cart_cross_sell_col' ) );

		return $columns;
	}
}

/**
 * ========================================================================================================
 * Products CheckOut Page
 * ========================================================================================================
 */

if ( ! function_exists( 'nevo_checkout_coupon_form' ) ) {
	function nevo_checkout_coupon_form() {
		if ( ! wc_coupons_enabled() ) { // @codingStandardsIgnoreLine.
			return;
		}
		?>
		<div class="woocommerce-form-coupon-toggle">
			<?php wc_print_notice( apply_filters( 'woocommerce_checkout_coupon_message', esc_html__( 'Have a coupon?', 'nevo' ) . ' <a href="#" class="showcoupon">' . esc_html__( 'Click here to enter your code', 'nevo' ) . '</a>' ), 'notice' ); ?>
		</div>

		<form class="checkout_coupon woocommerce-form-coupon cv-mb-4" method="post" style="display:none">

			<p><?php esc_html_e( 'If you have a coupon code, please apply it below.', 'nevo' ); ?></p>

			<div class="form-row flex-row d-flex align-items-center justify-content-between flex-nowrap cv-col-12">
				<div class="flex-col flex-grow">
					<input type="text" name="coupon_code" class="input-text cv-col-12" placeholder="<?php esc_attr_e( 'Coupon code', 'nevo' ); ?>" id="coupon_code" value="" />
				</div>
				<div class="flex-col">
					<button type="submit" class="button" name="apply_coupon" value="<?php esc_attr_e( 'Apply coupon', 'nevo' ); ?>"><?php esc_html_e( 'Apply coupon', 'nevo' ); ?></button>
				</div>
			</div>

			<div class="clear"></div>
		</form>
		<?php 
	}
}

/**
 * ========================================================================================================
 * Products MY Account Page
 * ========================================================================================================
 */

if ( ! function_exists( 'nevo_override_woocommerce_account_navigation' ) ) {
	/**
	 * Woocommerce account navagation
	 */
	function nevo_override_woocommerce_account_navigation() {
		
		do_action( 'woocommerce_before_account_navigation' );
		$icons = array(
			'dashboard' => 'bi-inbox',
			'orders' => 'bi-bag',
			'downloads' => 'bi-cloud-download',
			'edit-address' => 'bi-geo-alt',
			'edit-account' => 'bi-person',
			'customer-logout' => 'bi-arrow-return-right',
			'my-licenses' => 'bi-key',
		);
		$default_icon = 'bi-info';
		$text = '';
		$user = wp_get_current_user();
		$size  = '96';
		if ( is_user_logged_in() && $user ) {
			$avatar = get_avatar_url( $user->ID );
			$text = $user->display_name;
			if ( ! $text ) {
				$text = $user->user_login;
			}
		} else {
			$avatar = get_template_directory_uri() . '/assets/images/avatar.jpg';
			$text 	= esc_html__( 'Login', 'nevo' );
		}
		?>
		<nav class="woocommerce-MyAccount-navigation">
			<div class="nevo-header-account text-center">
				<div class="user-avatar d-flex justify-content-center">
					<img class="rounded-circle d-block" height="<?php echo esc_attr( $size ); ?>" width="<?php echo esc_attr( $size ); ?>" src="<?php echo esc_url( $avatar ); ?>" alt="" style="height:<?php echo esc_attr( $size ); ?>px;">
				</div>
				<span class="user-name text-capitalize d-block"><?php echo $text; // WPCS: XSS ok. ?></span>
			</div>
			<ul class="list-inline">
				<?php foreach ( wc_get_account_menu_items() as $endpoint => $label ) : ?>
					<li class="border-top <?php echo wc_get_account_menu_item_classes( $endpoint ); ?>">
						<a class="d-flex align-items-center" href="<?php echo esc_url( wc_get_account_endpoint_url( $endpoint ) ); ?>">
							<?php $icon = isset( $icons[ $endpoint ] ) ? $icons[ $endpoint ] : $default_icon; ?>
							<span class="nevo-icon" aria-hidden="true"><?php echo nevo_get_svg_icon( 'ui', $icon, 20 ); ?></span>
							<span class="info-text"><?php echo esc_html( $label ); ?></span>
						</a>
					</li>
				<?php endforeach; ?>
			</ul>
		</nav>
		<?php do_action( 'woocommerce_after_account_navigation' );
	}
}

/**
 * ========================================================================================================
 * Legacy WooCommerce columns filter.
 * ========================================================================================================
 */
if ( ! function_exists( 'nevo_product_columns_wrapper' ) ) {
	/**
	 * Product columns wrapper
	 *
	 * @since   1.0
	 * @return  void
	 */
	function nevo_product_columns_wrapper() {
		$columns = nevo_loop_columns();
		echo '<div class="columns-' . absint( $columns ) . '">';
	}
}

if ( ! function_exists( 'nevo_loop_columns' ) ) {
	/**
	 * Default loop columns on product archives
	 *
	 * @return integer products per row
	 * @since  1.0.0
	 */
	function nevo_loop_columns() {
		$columns = wc_get_loop_prop( 'columns' );

		if ( function_exists( 'wc_get_default_products_per_row' ) ) {
			$columns = wc_get_default_products_per_row();
		}

		return apply_filters( 'nevo_loop_columns', $columns );
	}
}

if ( ! function_exists( 'nevo_product_columns_wrapper_close' ) ) {
	/**
	 * Product columns wrapper close
	 *
	 * @since   1.0
	 * @return  void
	 */
	function nevo_product_columns_wrapper_close() {
		echo '</div>';
	}
}

/**
 * ========================================================================================================
 * Global Products
 * ========================================================================================================
 */
if ( ! function_exists( 'nevo_before_quantity_input_field' ) ) {
	/**
	 * Product add Before Quantity input
	 *
	 * @since   1.0
	 * @return  void
	 */
	function nevo_before_quantity_input_field() {
		echo '<span class="product-qty minus" data-qty="minus">-</span>';
	}
}

if ( ! function_exists( 'nevo_after_quantity_input_field' ) ) {
	/**
	 * Product add after Quantity input
	 *
	 * @since   1.0
	 * @return  void
	 */
	function nevo_after_quantity_input_field() {
		echo '<span class="product-qty plus" data-qty="plus">+</span>';
	}
}

if ( ! function_exists( 'nevo_image_alt' ) ) {

	/**
	 * Get image alt
	 *
	 * @param bolean $id The image id.
	 * @param string $alt The alternate.
	 * @param bolean $placeholder The bolean.
	 *
	 * @return     string  The image alt
	 */
	function nevo_image_alt( $id = null, $alt = '', $placeholder = false ) {
		if ( ! $id ) {
			if ( $placeholder ) {
				return esc_attr__( 'Placeholder image', 'nevo' );
			}

			return esc_attr__( 'Error image', 'nevo' );
		}

		$data    = get_post_meta( $id, '_wp_attachment_image_alt', true );
		$img_alt = ! empty( $data ) ? $data : $alt;

		return $img_alt;
	}
}

if ( ! function_exists( 'nevo_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments
	 * Ensure cart contents update when products are added to the cart via AJAX
	 *
	 * @param  array $fragments Fragments to refresh via AJAX.
	 * @return array            Fragments to refresh via AJAX
	 */
	function nevo_cart_link_fragment( $fragments ) {
		//global $woocommerce;
		$sub_total = WC()->cart->get_cart_subtotal();
		$quantities = WC()->cart->get_cart_item_quantities();
		
		$fragments['.nevo-cart-subtotal'] = '<span class="nevo-cart-subtotal">' . $sub_total . '</span>';
		
		$qty   = array_sum( $quantities );
		$fragments['.cart-count'] = '<span class="cart-count has-warning-background-color">' . $qty . '</span>';
		
		ob_start();
		nevo_handheld_footer_bar_cart_link();
		$fragments['a.footer-cart-contents'] = ob_get_clean();
		
		return $fragments;
	}
}

if ( ! function_exists( 'nevo_custom_product_search_form' ) ) {
	/**
	 * Custom product search form
	 *
	 * @return mixed
	 */
	function nevo_custom_product_search_form() {
		global $product_search_form_index;

		if ( empty( $product_search_form_index ) ) {
			$product_search_form_index = 0;
		}
		$index = $product_search_form_index++;

		?>
		<form method="get" class="search-form d-flex align-items-center flex-nowrap cv-col-12 woocommerce-product-search" action="<?php echo esc_url( home_url( '/' ) ); ?>">
			<label class="cv-col-12">
				<span class="screen-reader-text" for="woocommerce-product-search-field-<?php echo isset( $index ) ? absint( $index ) : 0; ?>"><?php echo apply_filters( 'nevo_search_label', _x( 'Search for:', 'label', 'nevo' ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></span>
				<input type="search" id="woocommerce-product-search-field-<?php echo isset( $index ) ? absint( $index ) : 0; ?>" class="search-field cv-col-12" placeholder="<?php echo esc_attr( apply_filters( 'nevo_search_product_placeholder', _x( 'Search products &hellip;', 'placeholder', 'nevo' ) ) ); ?>" value="<?php echo esc_attr( get_search_query() ); ?>" name="s">
			</label>
			<?php
			if ( apply_filters( 'nevo_search_button_icon', true ) ) {
				printf(
					'<button type="submit" value="%1$s" class="search-submit" aria-label="%1$s">%2$s</button>',
					esc_attr( apply_filters( 'nevo_search_button', _x( 'Search', 'submit button', 'nevo' ) ) ),
					'<span class="nevo-icon">'. nevo_get_svg_icon( 'ui', 'bi-search', '', 20 ) .'</span>' // phpcs:ignore -- Escaping not necessary here.
				);
			} else {
				printf(
					'<input type="submit" class="search-submit" value="%s">',
					apply_filters( 'nevo_search_button', _x( 'Search', 'submit button', 'nevo' ) ) // phpcs:ignore -- Escaping not necessary here.
				);
			}
			?>
			<input type="hidden" name="post_type" value="product" />
		</form>
		<?php
	}
}

/**
 * ========================================================================================================
 * SECTION Products Content
 * ========================================================================================================
 */
if ( ! function_exists( 'nevo_product_categories' ) ) {
	/**
	 * Display Product Categories
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function nevo_product_categories( $args ) {
		$args = apply_filters(
			'nevo_product_categories_args',
			array(
				'limit'            => 4,
				'columns'          => 4,
				'child_categories' => 0,
				'orderby'          => 'menu_order', //Mặc định là sắp xếp theo “tên”, có thể được đặt thành “id”, “slug” hoặc “menu_order”. Nếu bạn muốn đặt hàng theo id bạn đã chỉ định thì bạn có thể sử dụng orderby="include"
				'title'            => __( 'Shop by Category', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'product_categories',
			apply_filters(
				'nevo_product_categories_shortcode_args',
				array(
					'number'  => intval( $args['limit'] ),
					'columns' => intval( $args['columns'] ),
					'orderby' => esc_attr( $args['orderby'] ),
					'parent'  => esc_attr( $args['child_categories'] ),
					//'ids' => '49, 50, 51',
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns product categories
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product-category' ) ) {
			echo '<section class="nevo-product-section nevo-product-categories" aria-label="' . esc_attr__( 'Product Categories', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_product_categories' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_product_categories_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_product_categories' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_recent_products' ) ) {
	/**
	 * Display Recent Products
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function nevo_recent_products( $args ) {
		$args = apply_filters(
			'nevo_recent_products_args',
			array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'date',
				'order'   => 'desc',
				'title'   => __( 'New In', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'products',
			apply_filters(
				'nevo_recent_products_shortcode_args',
				array(
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="nevo-product-section nevo-recent-products" aria-label="' . esc_attr__( 'Recent Products', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_recent_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_recent_products_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_recent_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_featured_products' ) ) {
	/**
	 * Display Featured Products
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function nevo_featured_products( $args ) {
		$args = apply_filters(
			'nevo_featured_products_args',
			array(
				'limit'      => 4,
				'columns'    => 4,
				'orderby'    => 'date',
				'order'      => 'desc',
				'visibility' => 'featured',
				'title'      => __( 'We Recommend', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'products',
			apply_filters(
				'nevo_featured_products_shortcode_args',
				array(
					'per_page'   => intval( $args['limit'] ),
					'columns'    => intval( $args['columns'] ),
					'orderby'    => esc_attr( $args['orderby'] ),
					'order'      => esc_attr( $args['order'] ),
					'visibility' => esc_attr( $args['visibility'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="nevo-product-section nevo-featured-products" aria-label="' . esc_attr__( 'Featured Products', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_featured_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_featured_products_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_featured_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_popular_products' ) ) {
	/**
	 * Display Popular Products
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function nevo_popular_products( $args ) {
		$args = apply_filters(
			'nevo_popular_products_args',
			array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'rating',
				'order'   => 'desc',
				'title'   => __( 'Fan Favorites', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'products',
			apply_filters(
				'nevo_popular_products_shortcode_args',
				array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="nevo-product-section nevo-popular-products" aria-label="' . esc_attr__( 'Popular Products', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_popular_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_popular_products_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_popular_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_on_sale_products' ) ) {
	/**
	 * Display On Sale Products
	 *
	 * @param array $args the product section args.
	 * @since  1.0.0
	 * @return void
	 */
	function nevo_on_sale_products( $args ) {
		$args = apply_filters(
			'nevo_on_sale_products_args',
			array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'date',
				'order'   => 'desc',
				'on_sale' => 'true',
				'title'   => __( 'On Sale', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'products',
			apply_filters(
				'nevo_on_sale_products_shortcode_args',
				array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
					'on_sale'  => esc_attr( $args['on_sale'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="nevo-product-section nevo-on-sale-products" aria-label="' . esc_attr__( 'On Sale Products', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_on_sale_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_on_sale_products_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_on_sale_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_best_selling_products' ) ) {
	/**
	 * Display Best Selling Products
	 *
	 * @since 1.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function nevo_best_selling_products( $args ) {
		$args = apply_filters(
			'nevo_best_selling_products_args',
			array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'popularity',
				'order'   => 'desc',
				'title'   => esc_attr__( 'Best Sellers', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'products',
			apply_filters(
				'nevo_best_selling_products_shortcode_args',
				array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== $shortcode_content && false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="nevo-product-section nevo-best-selling-products" aria-label="' . esc_attr__( 'Best Selling Products', 'nevo' ) . '">';

			do_action( 'nevo_homepage_before_best_selling_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'nevo_homepage_after_best_selling_products_title' );

			echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			do_action( 'nevo_homepage_after_best_selling_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'nevo_promoted_products' ) ) {
	/**
	 * Featured and On-Sale Products
	 * Check for featured products then on-sale products and use the appropiate shortcode.
	 * If neither exist, it can fallback to show recently added products.
	 *
	 * @since  1.0
	 * @param integer $per_page total products to display.
	 * @param integer $columns columns to arrange products in to.
	 * @param boolean $recent_fallback Should the function display recent products as a fallback when there are no featured or on-sale products?.
	 * @uses  nevo_is_woocommerce_activated()
	 * @uses  wc_get_featured_product_ids()
	 * @uses  wc_get_product_ids_on_sale()
	 * @uses  nevo_do_shortcode()
	 * @return void
	 */
	function nevo_promoted_products( $per_page = '2', $columns = '2', $recent_fallback = true ) {
		if ( Nevo()->is_woocommerce_active() ) {

			if ( wc_get_featured_product_ids() ) {

				echo '<h2>' . esc_html__( 'Featured Products', 'nevo' ) . '</h2>';

				// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
				echo nevo_do_shortcode(
					'featured_products',
					array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				);
				// phpcs:enable
			} elseif ( wc_get_product_ids_on_sale() ) {

				echo '<h2>' . esc_html__( 'On Sale Now', 'nevo' ) . '</h2>';

				// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
				echo nevo_do_shortcode(
					'sale_products',
					array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				);
				// phpcs:enable
			} elseif ( $recent_fallback ) {

				echo '<h2>' . esc_html__( 'New In Store', 'nevo' ) . '</h2>';

				// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
				echo nevo_do_shortcode(
					'recent_products',
					array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				);
				// phpcs:enable
			}
		}
	}
}

if ( ! function_exists( 'nevo_handheld_footer_bar' ) ) {
	/**
	 * Display a menu intended for use on handheld devices
	 *
	 * @since 1.0
	 */
	function nevo_handheld_footer_bar() {
		$links = array(
			'my-account' => array(
				'priority' => 10,
				'callback' => 'nevo_handheld_footer_bar_account_link',
			),
			'cart'       => array(
				'priority' => 30,
				'callback' => 'nevo_handheld_footer_bar_cart_link',
			),
		);

		if ( did_action( 'woocommerce_blocks_enqueue_cart_block_scripts_after' ) || did_action( 'woocommerce_blocks_enqueue_checkout_block_scripts_after' ) ) {
			return;
		}

		if ( wc_get_page_id( 'myaccount' ) === -1 ) {
			unset( $links['my-account'] );
		}

		if ( wc_get_page_id( 'cart' ) === -1 ) {
			unset( $links['cart'] );
		}

		$links = apply_filters( 'nevo_handheld_footer_bar_links', $links );
		?>
		<div class="nevo-handheld-footer-bar">
			<ul class="columns-<?php echo count( $links ); ?> row-cols row-cols-<?php echo count( $links ); ?> align-items-center list-inline text-center">
				<?php foreach ( $links as $key => $link ) : ?>
					<li class="<?php echo esc_attr( $key ); ?>">
						<?php
						if ( $link['callback'] ) {
							call_user_func( $link['callback'], $key, $link );
						}
						?>
					</li>
				<?php endforeach; ?>
			</ul>
		</div>
		<?php
	}
}

if ( ! function_exists( 'nevo_handheld_footer_bar_cart_link' ) ) {
	/**
	 * The cart callback function for the handheld footer bar
	 *
	 * @since 1.0
	 */
	function nevo_handheld_footer_bar_cart_link() {
		if ( ! nevo_woo_cart_available() ) {
			return;
		}
		?>
			<a class="footer-cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>"><?php esc_html_e( 'Cart', 'nevo' ); ?>
				<span class="count bg-info"><?php echo wp_kses_data( WC()->cart->get_cart_contents_count() ); ?></span>
			</a>
		<?php
	}
}

if ( ! function_exists( 'nevo_handheld_footer_bar_account_link' ) ) {
	/**
	 * The account callback function for the handheld footer bar
	 *
	 * @since 1.0
	 */
	function nevo_handheld_footer_bar_account_link() {
		echo '<a href="' . esc_url( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . '">' . esc_attr__( 'My Account', 'nevo' ) . '</a>';
	}
}

if ( ! function_exists( 'nevo_single_product_pagination' ) ) {
	/**
	 * Single Product Pagination
	 *
	 * @since 1.0
	 */
	function nevo_single_product_pagination() {
		if ( class_exists( 'Nevo_Product_Pagination' ) || true !== get_theme_mod( 'nevo_product_pagination' ) ) {
			return;
		}

		// Show only products in the same category?
		$in_same_term   = apply_filters( 'nevo_single_product_pagination_same_category', true );
		$excluded_terms = apply_filters( 'nevo_single_product_pagination_excluded_terms', '' );
		$taxonomy       = apply_filters( 'nevo_single_product_pagination_taxonomy', 'product_cat' );

		$previous_product = nevo_get_previous_product( $in_same_term, $excluded_terms, $taxonomy );
		$next_product     = nevo_get_next_product( $in_same_term, $excluded_terms, $taxonomy );

		if ( ! $previous_product && ! $next_product ) {
			return;
		}

		?>
		<nav class="nevo-product-pagination" aria-label="<?php esc_attr_e( 'More products', 'nevo' ); ?>">
			<?php if ( $previous_product ) : ?>
				<a href="<?php echo esc_url( $previous_product->get_permalink() ); ?>" rel="prev">
					<?php echo wp_kses_post( $previous_product->get_image() ); ?>
					<span class="nevo-product-pagination__title"><?php echo wp_kses_post( $previous_product->get_name() ); ?></span>
				</a>
			<?php endif; ?>

			<?php if ( $next_product ) : ?>
				<a href="<?php echo esc_url( $next_product->get_permalink() ); ?>" rel="next">
					<?php echo wp_kses_post( $next_product->get_image() ); ?>
					<span class="nevo-product-pagination__title"><?php echo wp_kses_post( $next_product->get_name() ); ?></span>
				</a>
			<?php endif; ?>
		</nav><!-- .nevo-product-pagination -->
		<?php
	}
}

if ( ! function_exists( 'nevo_sticky_single_add_to_cart' ) ) {
	/**
	 * Sticky Add to Cart
	 *
	 * @since 1.0
	 */
	function nevo_sticky_single_add_to_cart() {
		global $product;

		if ( class_exists( 'Nevo_Sticky_Add_to_Cart' ) ) {
			return;
		}

		if ( ! $product || ! is_product() ) {
			return;
		}

		$show = false;

		if ( $product->is_purchasable() && $product->is_in_stock() ) {
			$show = true;
		} elseif ( $product->is_type( 'external' ) ) {
			$show = true;
		}

		if ( ! $show ) {
			return;
		}

		$params = apply_filters(
			'nevo_sticky_add_to_cart_params',
			array(
				'trigger_class' => 'entry-summary',
			)
		);

		wp_localize_script( 'nevo-sticky-add-to-cart', 'nevo_sticky_add_to_cart_params', $params );

		wp_enqueue_script( 'nevo-sticky-add-to-cart' );
		?>
			<section class="nevo-sticky-add-to-cart">
				<div class="nevo-container">
					<div class="nevo-sticky-add-to-cart__content">
						<?php echo wp_kses_post( woocommerce_get_product_thumbnail() ); ?>
						<div class="nevo-sticky-add-to-cart__content-product-info">
							<span class="nevo-sticky-add-to-cart__content-title"><?php esc_html_e( 'You\'re viewing:', 'nevo' ); ?> <strong><?php the_title(); ?></strong></span>
							<span class="nevo-sticky-add-to-cart__content-price"><?php echo wp_kses_post( $product->get_price_html() ); ?></span>
							<?php echo wp_kses_post( wc_get_rating_html( $product->get_average_rating() ) ); ?>
						</div>
						<a href="<?php echo esc_url( $product->add_to_cart_url() ); ?>" class="nevo-sticky-add-to-cart__content-button button alt" rel="nofollow">
							<?php echo esc_attr( $product->add_to_cart_text() ); ?>
						</a>
					</div>
				</div>
			</section><!-- .nevo-sticky-add-to-cart -->
		<?php
	}
}

if ( ! function_exists( 'nevo_woocommerce_brands_homepage_section' ) ) {
	/**
	 * Display WooCommerce Brands.
	 * Requires WooCommerce Brands.
	 *
	 * @since  1.0
	 * @link   https://woocommerce.com/products/brands/
	 * @uses   apply_filters()
	 * @uses   nevo_do_shortcode()
	 * @uses   wp_kses_post()
	 * @uses   do_action()
	 * @return void
	 */
	function nevo_woocommerce_brands_homepage_section() {
		$args = apply_filters(
			'nevo_woocommerce_brands_args',
			array(
				'number'     => 6,
				'columns'    => 4,
				'orderby'    => 'name',
				'show_empty' => false,
				'title'      => __( 'Shop by Brand', 'nevo' ),
			)
		);

		$shortcode_content = nevo_do_shortcode(
			'product_brand_thumbnails',
			apply_filters(
				'nevo_woocommerce_brands_shortcode_args',
				array(
					'number'     => absint( $args['number'] ),
					'columns'    => absint( $args['columns'] ),
					'orderby'    => esc_attr( $args['orderby'] ),
					'show_empty' => (bool) $args['show_empty'],
				)
			)
		);

		echo '<section class="nevo-product-section nevo-woocommerce-brands" aria-label="' . esc_attr__( 'Product Brands', 'nevo' ) . '">';

		do_action( 'nevo_homepage_before_woocommerce_brands' );

		echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

		do_action( 'nevo_homepage_after_woocommerce_brands_title' );

		echo $shortcode_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		do_action( 'nevo_homepage_after_woocommerce_brands' );

		echo '</section>';
	}
}

if ( ! function_exists( 'nevo_woocommerce_brands_archive' ) ) {
	/**
	 * Display brand image on brand archives
	 * Requires WooCommerce Brands.
	 *
	 * @since  1.0
	 * @link   https://woocommerce.com/products/brands/
	 * @uses   is_tax()
	 * @uses   wp_kses_post()
	 * @uses   get_brand_thumbnail_image()
	 * @uses   get_queried_object()
	 * @return void
	 */
	function nevo_woocommerce_brands_archive() {
		if ( is_tax( 'product_brand' ) ) {
			echo wp_kses_post( get_brand_thumbnail_image( get_queried_object() ) );
		}
	}
}

if ( ! function_exists( 'nevo_woocommerce_brands_single' ) ) {
	/**
	 * Output product brand image for use on single product pages
	 * Requires WooCommerce Brands.
	 *
	 * @since  1.0
	 * @link   https://woocommerce.com/products/brands/
	 * @uses   nevo_do_shortcode()
	 * @uses   wp_kses_post()
	 * @return void
	 */
	function nevo_woocommerce_brands_single() {
		$brand = nevo_do_shortcode(
			'product_brand',
			array(
				'class' => '',
			)
		);

		if ( empty( $brand ) ) {
			return;
		}

		?>
		<div class="nevo-wc-brands-single-product">
			<?php echo wp_kses_post( $brand ); ?>
		</div>
		<?php
	}
}
